<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Interest;
use Illuminate\Support\Facades\Auth;
use App\Models\UserInterest;
use App\Models\Chat;

class UserController extends Controller
{
    
    
    public function login(Request $request)
    {
        // Check if the user is already logged in (i.e., the auth_token is in the session)
        if ($request->session()->has('auth_token')) {
            // If the auth_token exists in the session, redirect the user to the home page
            return redirect()->route('home');
        }
    
        // If not logged in, show the login page
        return view('web.login');
    }
    

    public function sendotp(Request $request)
    {

        $validated = $request->validate([
            'username' => 'required|string|max:255',
            'phone_number' => 'required|string|regex:/^([0-9\s\-\+\(\)]*)$/',
        ]);

        $username = $validated['username'];
        $phone_number = $validated['phone_number'];

        // Check if the user exists
        $user = User::where('number', $phone_number)->orWhere('name', $username)->first();

        if ($user) {
            // User exists, update OTP
            $user->otp = '123456'; // Static OTP
            $user->save();

            // Return OTP popup (you can pass a success message here)
            return response()->json(['status' => 'otp_sent', 'message' => 'OTP sent.']);
        } else {
            // User does not exist, create new user with OTP
            $user = User::create([
                'name' => $username,
                'number' => $phone_number,
                'otp' => '123456', // Static OTP
            ]);

            // Return OTP popup
            return response()->json(['status' => 'otp_sent', 'message' => 'OTP sent.']);
        }
    }

    public function verifyOtp(Request $request)
    {
        $validated = $request->validate([
            'otp' => 'required|string|size:6',
            'phone_number' => 'required',
        ]);
    
        // Find the user by phone number
        $user = User::where('number', $validated['phone_number'])->first();
    
        if ($user && $user->otp == $validated['otp']) {
            // Generate a unique token
            $token = bin2hex(random_bytes(16));
    
            // Store the token in the database
            $user->auth_token = $token;
            $user->save();
    
            // Store the token in the session for future requests
            session(['auth_token' => $token]);
    
            return response()->json([
                'status' => 'success',
                'message' => 'OTP verified.',
                'token' => $token,
            ]);
        } else {
            return response()->json(['status' => 'error', 'message' => 'Invalid OTP. Please try again.'], 422);
        }
    }
    
    


    public function home(Request $request)  
    {
        $authToken = session('auth_token');
        $user = User::where('auth_token', $authToken)->first();
        
        return view('web.userhome');
    }

    public function chat(Request $request)
{
    $authToken = session('auth_token');
    $user = User::where('auth_token', $authToken)->first();

    $type = $request->query('type', 'yourself'); // Default to 'yourself' if no type is provided

    return view('web.chat', compact('user', 'type'));
}



public function getInterests(Request $request)
{
    $type = $request->query('type', null); // Get the 'type' from the query parameters

    // Fetch interests based on the type
    $interests = Interest::when($type, function ($query, $type) {
        return $query->where('type', $type);
    })->get();

    return response()->json($interests); // Return the filtered interests as JSON
}



  // Controller Method
  public function store(Request $request)
  {
      $request->validate([
          'interests' => 'required|string',
      ]);
  
      $authToken = session('auth_token');
      $user = User::where('auth_token', $authToken)->first();
  
      if (!$user) {
          return response()->json(['error' => 'User not found'], 404);
      }
  
      $userInterest = UserInterest::updateOrCreate(
          ['user_id' => $user->id],
          [
              'interests' => $request->interests,
              'active_status' => 1,
          ]
      );
  
      $inputInterests = array_map('trim', explode(',', $request->interests));
  
      // Get all users with their interests
      $matchingUsersQuery = UserInterest::with('user')
          ->where('user_id', '!=', $user->id)
          ->get();
  
      // Process matching users
      $matchingUsers = collect();
      foreach ($matchingUsersQuery as $userInterest) {
          $theirInterests = array_map('trim', explode(',', $userInterest->interests));
          $commonInterests = array_values(array_intersect($inputInterests, $theirInterests));
          
          if (count($commonInterests) > 0) {
              $matchingUsers->push([
                  'id' => $userInterest->user->id,
                  'name' => $userInterest->user->name,
                  'number' => $userInterest->user->number,
                  'common_interests' => $commonInterests,
                  'common_count' => count($commonInterests)
              ]);
          }
      }
  
      // Sort by common interest count
      $matchingUsers = $matchingUsers->sortByDesc('common_count')->values();
  
      // Get remaining users who don't have matching interests
      $matchingUserIds = $matchingUsers->pluck('id')->toArray();
      $remainingUsers = User::where('id', '!=', $user->id)
          ->whereNotIn('id', $matchingUserIds)
          ->get()
          ->map(function ($user) {
              return [
                  'id' => $user->id,
                  'name' => $user->name,
                  'number' => $user->number,
                  'common_interests' => [],
                  'common_count' => 0
              ];
          });
  
      // Combine matching and remaining users
      $finalUserList = $matchingUsers->concat($remainingUsers);
  
      return response()->json([
          'message' => 'Interests saved successfully',
          'matching_users' => $finalUserList->values()
      ]);
  }
    
    



    public function updateActiveStatus(Request $request)
    {
        $authToken = session('auth_token');
    
       
        $user = User::where('auth_token', $authToken)->first();
    
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }
    
        try {
           
            UserInterest::where('user_id', $user->id)
                ->update(['active_status' => 0]);
    
           
            Chat::where('user_id', $user->id)->delete();
    
            return response()->json([
                'success' => true,
                'message' => 'Active status and chats updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update active status and delete chats',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    

}



