import { AxiosRequestConfig } from 'axios';
import type { SessionDescription } from 'sdp-parser2';
import { default as UAParser_2 } from 'ua-parser-js';

export declare enum AgoraAPIName {
    CREATE_CLIENT = "createClient",
    CHECK_SYSTEM_REQUIREMENTS = "checkSystemRequirements",
    SET_AREA = "setArea",
    PRELOAD = "PRELOAD",
    /**
     * Track checker
     */
    CHECK_VIDEO_TRACK_IS_ACTIVE = "checkVideoTrackIsActive",
    CHECK_AUDIO_TRACK_IS_ACTIVE = "checkAudioTrackIsActive",
    /**
     * Track Creator
     */
    CREATE_MIC_AUDIO_TRACK = "createMicrophoneAudioTrack",
    CREATE_CUSTOM_AUDIO_TRACK = "createCustomAudioTrack",
    CREATE_BUFFER_AUDIO_TRACK = "createBufferSourceAudioTrack",
    CREATE_CAM_VIDEO_TRACK = "createCameraVideoTrack",
    CREATE_CUSTOM_VIDEO_TRACK = "createCustomVideoTrack",
    CREATE_MIC_AND_CAM_TRACKS = "createMicrophoneAndCameraTracks",
    CREATE_SCREEN_VIDEO_TRACK = "createScreenVideoTrack",
    /**
     * Client 相关
     */
    SET_ENCRYPTION_CONFIG = "Client.setEncryptionConfig",
    START_PROXY_SERVER = "Client.startProxyServer",
    STOP_PROXY_SERVER = "Client.stopProxyServer",
    SET_PROXY_SERVER = "Client.setProxyServer",
    SET_TURN_SERVER = "Client.setTurnServer",
    SET_CLIENT_ROLE = "Client.setClientRole",
    SET_LOW_STREAM_PARAMETER = "Client.setLowStreamParameter",
    ENABLE_DUAL_STREAM = "Client.enableDualStream",
    DISABLE_DUAL_STREAM = "Client.disableDualStream",
    JOIN = "Client.join",
    LEAVE = "Client.leave",
    PUBLISH = "Client.publish",
    UNPUBLISH = "Client.unpublish",
    SUBSCRIBE = "Client.subscribe",
    MASS_SUBSCRIBE = "Client.massSubscribe",
    MASS_UNSUBSCRIBE = "Client.massUnsubscribe",
    UNSUBSCRIBE = "Client.unsubscribe",
    RENEW_TOKEN = "Client.renewToken",
    SET_REMOTE_VIDEO_STREAM_TYPE = "Client.setRemoteVideoStreamType",
    SET_STREAM_FALLBACK_OPTION = "Client.setStreamFallbackOption",
    ENABLE_AUDIO_VOLUME_INDICATOR = "Client.enableAudioVolumeIndicator",
    SEND_CUSTOM_REPORT_MESSAGE = "Client.sendCustomReportMessage",
    INSPECT_VIDEO_CONTENT = "Client.inspectVideoContent",
    STOP_INSPECT_VIDEO_CONTENT = "Client.stopInspectVideoContent",
    JOIN_FALLBACK_TO_PROXY = "Client._joinFallbackToProxy",
    /**
     * 推流相关
     */
    ON_LIVE_STREAM_WARNING = "Client.onLiveStreamWarning",
    ON_LIVE_STREAM_ERROR = "Client.onLiveStreamingError",
    START_LIVE_STREAMING = "Client.startLiveStreaming",
    SET_LIVE_TRANSCODING = "Client.setLiveTranscoding",
    STOP_LIVE_STREAMING = "Client.stopLiveStreaming",
    /**
     * 跨直播间连麦
     */
    START_CHANNEL_MEDIA_RELAY = "Client.startChannelMediaRelay",
    UPDATE_CHANNEL_MEDIA_RELAY = "Client.updateChannelMediaRelay",
    STOP_CHANNEL_MEDIA_RELAY = "Client.stopChannelMediaRelay",
    /**
     * 配置下发
     */
    REQUEST_CONFIG_DISTRIBUTE = "_config-distribute-request",
    SET_CONFIG_DISTRIBUTE = "_configDistribute",
    /**
     * Tracks
     */
    LOCAL_TRACK_SET_MUTED = "LocalTrack.setMute",
    LOCAL_AUDIO_TRACK_PLAY = "LocalAudioTrack.play",
    LOCAL_AUDIO_TRACK_PLAY_IN_ELEMENT = "LocalAudioTrack.playInElement",
    LOCAL_AUDIO_TRACK_STOP = "LocalAudioTrack.stop",
    LOCAL_AUDIO_TRACK_SET_VOLUME = "LocalAudioTrack.setVolume",
    MIC_AUDIO_TRACK_SET_DEVICE = "MicrophoneAudioTrack.setDevice",
    BUFFER_AUDIO_TRACK_START = "BufferSourceAudioTrack.startProcessAudioBuffer",
    BUFFER_AUDIO_TRACK_STOP = "BufferSourceAudioTrack.stopProcessAudioBuffer",
    BUFFER_AUDIO_TRACK_PAUSE = "BufferSourceAudioTrack.pauseProcessAudioBuffer",
    BUFFER_AUDIO_TRACK_RESUME = "BufferSourceAudioTrack.resumeProcessAudioBuffer",
    BUFFER_AUDIO_TRACK_SEEK = "BufferSourceAudioTrack.seekAudioBuffer",
    LOCAL_VIDEO_TRACK_PLAY = "LocalVideoTrack.play",
    LOCAL_VIDEO_TRACK_STOP = "LocalVideoTrack.stop",
    LOCAL_VIDEO_TRACK_GET_VIDEO_VISIBLE = "LocalVideoTrack.getVideoElementVisibleStatus",
    LOCAL_VIDEO_TRACK_BEAUTY = "LocalVideoTrack.setBeautyEffect",
    LOCAL_VIDEO_SEND_SEI_DATA = "LocalVideoTrack.sendSeiData",
    CAM_VIDEO_TRACK_SET_DEVICE = "CameraVideoTrack.setDevice",
    CAM_VIDEO_TRACK_SET_ENCODER_CONFIG = "CameraVideoTrack.setEncoderConfiguration",
    REMOTE_VIDEO_TRACK_PLAY = "RemoteVideoTrack.play",
    REMOTE_VIDEO_TRACK_STOP = "RemoteVideoTrack.stop",
    REMOTE_VIDEO_TRACK_GET_VIDEO_VISIBLE = "RemoteVideoTrack.getVideoElementVisibleStatus",
    REMOTE_AUDIO_TRACK_PLAY = "RemoteAudioTrack.play",
    REMOTE_AUDIO_TRACK_STOP = "RemoteAudioTrack.stop",
    REMOTE_AUDIO_SET_VOLUME = "RemoteAudioTrack.setVolume",
    REMOTE_AUDIO_SET_OUTPUT_DEVICE = "RemoteAudioTrack.setOutputDevice",
    GET_MEDIA_STREAM_TRACK = "Track.getMediaStreamTrack",
    /**
     * Others
     */
    STREAM_TYPE_CHANGE = "streamTypeChange",
    CONNECTION_STATE_CHANGE = "connectionStateChange",
    LOAD_CONFIG_FROM_LOCALSTORAGE = "loadConfigFromLocalStorage",
    /**
     * ImageModeration
     */
    IMAGE_MODERATION_UPLOAD = "imageModerationUpload",
    /**
     * pc connected 恢复推流
     */
    REPUB_AFTER_PC_CONNECTED = "repubAfterPCConnected"
}

export declare enum AgoraAPITag {
    TRACER = "tracer"
}

export declare interface AgoraAudioReceiverStats extends ReceiverStats {
    /**
     * **Note:** Firefox 无法获取这个值
     */
    codec?: string;
    /**
     * 范围为 0 ～ 1
     * **Note:** Firefox 无法获取这个值
     */
    outputLevel?: number;
    decodingCNG?: number;
    decodingCTN?: number;
    decodingCTSG?: number;
    decodingNormal?: number;
    decodingPLC?: number;
    decodingPLCCNG?: number;
    expandRate?: number;
    preemptiveExpandRate?: number;
    secondaryDecodedRate?: number;
    speechExpandRate?: number;
    preferredJitterBufferMs?: number;
    jitterBufferMs: number;
    jitterMs: number;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    accelerateRate?: number;
    receivedFrames: number;
    droppedFrames: number;
    bytes: number;
    packets: number;
    packetsLost: number;
    packetsDiscarded: number;
    packetLostRate: number;
    ssrc: number;
    concealedSamples: number;
    totalProcessingDelay?: number;
    jitterBufferEmittedCount?: number;
    avgProcessingDelayMs?: number;
}

export declare interface AgoraAudioSenderStats extends SenderStats {
    /**
     * **Note:** Firefox 无法获取这个值
     */
    codec?: string;
    /**
     * 范围为 0 ～ 1
     * **Note:** Firefox 无法获取这个值
     */
    inputLevel?: number;
    /**
     * **Note:** Firefox/Safari 无法获取这个值
     * 如果输入音源不是麦克风，这个值取不到
     */
    aecReturnLoss?: number;
    /**
     * **Note:** Firefox/Safari 无法获取这个值
     * 如果输入音源不是麦克风，这个值取不到
     */
    aecReturnLossEnhancement?: number;
    /**
     * 残余回声数据只有 Chrome legacy stats 才能拿到
     */
    residualEchoLikelihood?: number;
    residualEchoLikelihoodRecentMax?: number;
    bytes: number;
    packets: number;
    packetsLost: number;
    packetLostRate: number;
    ssrc: number;
    rttMs: number;
    jitterMs: number;
    timestamp?: DOMHighResTimeStamp;
}

export declare interface AgoraBitrateStats {
    actualEncoded: number;
    transmit: number;
    targetEncoded?: number;
    retransmit?: number;
}

/**
 * 试用范围 Chrome 58 ~ Chrome 76
 */
export declare class AgoraLegacyStatsFilter extends AgoraWebRTCStatsFilter {
    private _stats;
    /** 统计帧解码间隔用 */
    private lastDecodeVideoReceiverStats;
    protected updateStats(): Promise<void>;
    private processBandwidthStats;
    private processSSRCStats;
    private _getStats;
    private statsResponsesToObjects;
}

export declare interface AgoraPCStats {
    timestamp: number;
    /**
     * 只有 legacy stats 能取到这 2 个参数
     */
    bitrate: AgoraBitrateStats;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    sendBandwidth?: number;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    rtt?: number;
    selectedCandidatePair: SelectedCandidatePairStats;
    sendPacketLossRate: number;
    recvPacketLossRate: number;
    videoSend: AgoraVideoSenderStats[];
    audioSend: AgoraAudioSenderStats[];
    videoRecv: AgoraVideoReceiverStats[];
    audioRecv: AgoraAudioReceiverStats[];
    /**
     * 只有IOS 17.4 / Safari 17.4 以下才会有值
     */
    qualityLimitationReason?: QualityLimitationReason;
    updateInterval?: number;
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
export declare class AgoraRTCError extends Error implements IAgoraRTCError {
    readonly code: AgoraRTCErrorCode;
    readonly message: string;
    readonly data?: any;
    readonly name: string;
    constructor(code: AgoraRTCErrorCode, message?: string, data?: any);
    toString(): string;
    print(level?: "error" | "warning", logger?: any): AgoraRTCError;
    throw(logger?: any): never;
}

/**
 * AgoraSDK 抛出的错误的 Code
 * @ignore
 */
/** @en
 * @ignore
 */
export declare enum AgoraRTCErrorCode {
    /**
     * 所有用户无法处理的、非预期的错误都使用这个错误码
     */
    UNEXPECTED_ERROR = "UNEXPECTED_ERROR",
    /** 服务端返回了非预期的响应  */
    UNEXPECTED_RESPONSE = "UNEXPECTED_RESPONSE",
    TIMEOUT = "TIMEOUT",
    /** 非法参数 */
    INVALID_PARAMS = "INVALID_PARAMS",
    /** 当前设备不可读 */
    NOT_READABLE = "NOT_READABLE",
    /** 浏览器不支持 */
    NOT_SUPPORTED = "NOT_SUPPORTED",
    /** 非法操作，比如在加入房间之前发布 */
    INVALID_OPERATION = "INVALID_OPERATION",
    /** 操作中止，比如在加入房间的中途离开房间 */
    OPERATION_ABORTED = "OPERATION_ABORTED",
    /** 安全策略限制 */
    WEB_SECURITY_RESTRICT = "WEB_SECURITY_RESTRICT",
    /** P2P信令交互过程中出现异常 */
    EXCHANGE_SDP_FAILED = "EXCHANGE_SDP_FAILED",
    /** P2P添加候选人过程中出现异常 */
    ADD_CANDIDATE_FAILED = "ADD_CANDIDATE_FAILED",
    /** DataChannel交互过程中出现异常 */
    DATACHANNEL_FAILED = "DATACHANNEL_FAILED",
    /**
     * http post 请求相关
     */
    NETWORK_ERROR = "NETWORK_ERROR",
    NETWORK_TIMEOUT = "NETWORK_TIMEOUT",
    NETWORK_RESPONSE_ERROR = "NETWORK_RESPONSE_ERROR",
    /**
     * report 相关
     */
    API_INVOKE_TIMEOUT = "API_INVOKE_TIMEOUT",
    /**
     * Device 模块相关
     */
    /** 枚举本地设备失败 */
    ENUMERATE_DEVICES_FAILED = "ENUMERATE_DEVICES_FAILED",
    /** 找不到指定设备 */
    DEVICE_NOT_FOUND = "DEVICE_NOT_FOUND",
    /**
     * Electron 相关
     */
    /** 无法获取 Electron 对象 */
    ELECTRON_IS_NULL = "ELECTRON_IS_NULL",
    /** 无法通过 Electron 获取屏幕共享源 */
    ELECTRON_DESKTOP_CAPTURER_GET_SOURCES_ERROR = "ELECTRON_DESKTOP_CAPTURER_GET_SOURCES_ERROR",
    /**
     * Stream 相关
     */
    /** init 因为其他的 init 操作而中止 */
    /** chrome 屏幕共享插件没有响应 */
    CHROME_PLUGIN_NO_RESPONSE = "CHROME_PLUGIN_NO_RESPONSE",
    /** chrome 屏幕共享插件没有安装 */
    CHROME_PLUGIN_NOT_INSTALL = "CHROME_PLUGIN_NOT_INSTALL",
    /** 媒体采集的参数不支持 */
    MEDIA_OPTION_INVALID = "MEDIA_OPTION_INVALID",
    /** 获取媒体设备权限被拒绝 */
    PERMISSION_DENIED = "PERMISSION_DENIED",
    /** 浏览器不支持该 constraint */
    CONSTRAINT_NOT_SATISFIED = "CONSTRAINT_NOT_SATISFIED",
    /** 播放时被浏览器自动播放策略阻止 */
    /** 创建小流时检测到大流没有视频轨 */
    /** 屏幕共享不允许使用大小流 */
    /** 轨道被禁用 */
    TRACK_IS_DISABLED = "TRACK_IS_DISABLED",
    /** 获取 video element 可见状态失败*/
    GET_VIDEO_ELEMENT_VISIBLE_ERROR = "GET_VIDEO_ELEMENT_VISIBLE_ERROR",
    /** 屏幕共享音频时用户没有点击 **分享音频** */
    SHARE_AUDIO_NOT_ALLOWED = "SHARE_AUDIO_NOT_ALLOWED",
    /** 使用RTCRtpEncodingParameters 进行小流编码失败*/
    LOW_STREAM_ENCODING_ERROR = "LOW_STREAM_ENCODING_ERROR",
    /** 设置 rtp encoding parameters 失败 */
    SET_ENCODING_PARAMETER_ERROR = "SET_ENCODING_PARAMETER_ERROR",
    /** Track 状态不可达 */
    TRACK_STATE_UNREACHABLE = "TRACK_STATE_UNREACHABLE",
    /**
     * Client join 相关
     */
    /** 用户提供的 Token 生成函数运行时出现错误 */
    /** 通过 string uid allocate 服务返回了非法的 int uid */
    INVALID_UINT_UID_FROM_STRING_UID = "INVALID_UINT_UID_FROM_STRING_UID",
    /** 尝试了数次均无法获取云代理服务 */
    CAN_NOT_GET_PROXY_SERVER = "CAN_NOT_GET_PROXY_SERVER",
    /** 尝试了数次均无法获取 gateway 地址 */
    CAN_NOT_GET_GATEWAY_SERVER = "CAN_NOT_GET_GATEWAY_SERVER",
    /** 从 AP 拿到的网关列表为空 (obsolete after 4.7.0)*/
    VOID_GATEWAY_ADDRESS = "VOID_GATEWAY_ADDRESS",
    /** UID 冲突，重复的 UID */
    UID_CONFLICT = "UID_CONFLICT",
    /** multi unilbs 服务，响应解析错误 */
    MULTI_UNILBS_RESPONSE_ERROR = "MULTI_UNILBS_RESPONSE_ERROR",
    /** 更新 Ticket 请求失败 */
    UPDATE_TICKET_FAILED = "UPDATE_TICKET_FAILED",
    /** join 过程中 TOKEN 过期了 */
    TOKEN_EXPIRE = "TOKEN_EXPIRE",
    /**
     * Client publish/unpublish 相关
     */
    /** 传入了非法的 local track */
    INVALID_LOCAL_TRACK = "INVALID_LOCAL_TRACK",
    /** 传入了非法的 track */
    INVALID_TRACK = "INVALID_TRACK",
    /** replaceTrack 等操作时找不到指定的 sender */
    SENDER_NOT_FOUND = "SENDER_NOT_FOUND",
    /** p2p 建立相关 */
    CREATE_OFFER_FAILED = "CREATE_OFFER_FAILED",
    SET_ANSWER_FAILED = "SET_ANSWER_FAILED",
    ICE_FAILED = "ICE_FAILED",
    PC_CLOSED = "PC_CLOSED",
    SENDER_REPLACE_FAILED = "SENDER_REPLACE_FAILED",
    /** 获取本地RTP能力失败 */
    GET_LOCAL_CAPABILITIES_FAILED = "GET_LOCAL_CAPABILITIES_FAILED",
    GET_LOCAL_CONNECTION_PARAMS_FAILED = "GET_LOCAL_CONNECTION_PARAMS_FAILED",
    SUBSCRIBE_FAILED = "SUBSCRIBE_FAILED",
    UNSUBSCRIBE_FAILED = "UNSUBSCRIBE_FAILED",
    /** 网关抛出的 P2P 断开 */
    GATEWAY_P2P_LOST = "GATEWAY_P2P_LOST",
    NO_ICE_CANDIDATE = "NO_ICE_CANDIDATE",
    /** 不允许发布多个视频轨道 */
    CAN_NOT_PUBLISH_MULTIPLE_VIDEO_TRACKS = "CAN_NOT_PUBLISH_MULTIPLE_VIDEO_TRACKS",
    EXIST_DISABLED_VIDEO_TRACK = "EXIST_DISABLED_VIDEO_TRACK",
    /**
     * Client subscribe/unsubscribe 相关
     */
    /** 找不到指定的订阅用户 */
    INVALID_REMOTE_USER = "INVALID_REMOTE_USER",
    /** 远端用户没有发布 */
    REMOTE_USER_IS_NOT_PUBLISHED = "REMOTE_USER_IS_NOT_PUBLISHED",
    /** 相同流的上一个订阅还没有完成 */
    /**
     * Client 其他
     */
    /** 自定义事件上报失败，通常是因为网络原因 */
    CUSTOM_REPORT_SEND_FAILED = "CUSTOM_REPORT_SEND_FAILED",
    /** 自定义上报太频繁 */
    CUSTOM_REPORT_FREQUENCY_TOO_HIGH = "CUSTOM_REPORT_FREQUENCY_TOO_HIGH",
    /**
     * Stream 混音相关
     */
    /** 下载在线音频文件失败 */
    FETCH_AUDIO_FILE_FAILED = "FETCH_AUDIO_FILE_FAILED",
    /** 读取本地 音频文件失败*/
    READ_LOCAL_AUDIO_FILE_ERROR = "READ_LOCAL_AUDIO_FILE_ERROR",
    /** 解码音频文件失败 */
    DECODE_AUDIO_FILE_FAILED = "DECODE_AUDIO_FILE_FAILED",
    /** 音效的 `soundID` 发生冲突 */
    /** 找不到指定的音效 ID */
    /**
     * Gateway 操作相关
     */
    /** 请求网关时 WS 断开 */
    WS_ABORT = "WS_ABORT",
    /** 请求网关前网关就已经断开 */
    WS_DISCONNECT = "WS_DISCONNECT",
    WS_ERR = "WS_ERR",
    /** p2p 拓展通道断开 */
    EXTERNAL_SIGNAL_ABORT = "EXTERNAL_SIGNAL_ABORT",
    /**
     * Live Streaming 相关
     */
    /** 无法和推流后台建立连接 */
    /** 推流任务已经存在 */
    LIVE_STREAMING_TASK_CONFLICT = "LIVE_STREAMING_TASK_CONFLICT",
    /** 推流/拉流参数错误 */
    LIVE_STREAMING_INVALID_ARGUMENT = "LIVE_STREAMING_INVALID_ARGUMENT",
    /** 推流/拉流服务内部错误 */
    LIVE_STREAMING_INTERNAL_SERVER_ERROR = "LIVE_STREAMING_INTERNAL_SERVER_ERROR",
    /** 推流 URL 被占用 */
    LIVE_STREAMING_PUBLISH_STREAM_NOT_AUTHORIZED = "LIVE_STREAMING_PUBLISH_STREAM_NOT_AUTHORIZED",
    /** 在非转码推流中调用了转码参数 */
    LIVE_STREAMING_TRANSCODING_NOT_SUPPORTED = "LIVE_STREAMING_TRANSCODING_NOT_SUPPORTED",
    /** 推流的目标 CDN 出现错误导致推流失败 */
    LIVE_STREAMING_CDN_ERROR = "LIVE_STREAMING_CDN_ERROR",
    /** 推流超时，请确认目标流是否存在 */
    LIVE_STREAMING_INVALID_RAW_STREAM = "LIVE_STREAMING_INVALID_RAW_STREAM",
    /** 推流超过 10 路流 */
    LIVE_STREAMING_WARN_STREAM_NUM_REACH_LIMIT = "LIVE_STREAMING_WARN_STREAM_NUM_REACH_LIMIT",
    /** 推流中的背景图片或者水印地址无法拉取(不影响推流流程) */
    LIVE_STREAMING_WARN_FAILED_LOAD_IMAGE = "LIVE_STREAMING_WARN_FAILED_LOAD_IMAGE",
    /** 推流请求太频繁（不影响推流流程） */
    LIVE_STREAMING_WARN_FREQUENT_REQUEST = "LIVE_STREAMING_WARN_FREQUENT_REQUEST",
    /**
     * WebGL/美颜相关
     */
    /** WebGL 内部错误 */
    WEBGL_INTERNAL_ERROR = "WEBGL_INTERNAL_ERROR",
    /** 美颜内部错误 */
    BEAUTY_PROCESSOR_INTERNAL_ERROR = "BEAUTY_PROCESSOR_INTERNAL_ERROR",
    /**
     * Cross Channel 相关
     */
    /** 等待 status 回调出错 */
    CROSS_CHANNEL_WAIT_STATUS_ERROR = "CROSS_CHANNEL_WAIT_STATUS_ERROR",
    /** 服务器加入源频道失败 */
    CROSS_CHANNEL_FAILED_JOIN_SRC = "CROSS_CHANNEL_FAILED_JOIN_SEC",
    /** 服务器加入目标频道失败 */
    CROSS_CHANNEL_FAILED_JOIN_DEST = "CROSS_CHANNEL_FAILED_JOIN_DEST",
    /** 源频道发送数据失败 */
    CROSS_CHANNEL_FAILED_PACKET_SENT_TO_DEST = "CROSS_CHANNEL_FAILED_PACKET_SENT_TO_DEST",
    /** 服务器回应出错 */
    CROSS_CHANNEL_SERVER_ERROR_RESPONSE = "CROSS_CHANNEL_SERVER_ERROR_RESPONSE",
    /**
     * AVC SEI 相关
     */
    /** 需要编码的 SEI 数据超过了最大大小 */
    METADATA_OUT_OF_RANGE = "METADATA_OUT_OF_RANGE",
    LOCAL_AEC_ERROR = "LOCAL_AEC_ERROR",
    /** 插件不合法 */
    INVALID_PLUGIN = "INVALID_PLUGIN",
    /** 抛出断开P2P的错误来让未执行完成的P2P操作结束*/
    DISCONNECT_P2P = "DISCONNECT_P2P",
    /** imageData转换为Blob时失败 */
    CONVERTING_IMAGEDATA_TO_BLOB_FAILED = "CONVERTING_IMAGEDATA_TO_BLOB_FAILED",
    CONVERTING_VIDEO_FRAME_TO_BLOB_FAILED = "CONVERTING_VIDEO_FRAME_TO_BLOB_FAILED",
    /** datachannel相关 */
    INIT_DATACHANNEL_TIMEOUT = "INIT_DATACHANNEL_TIMEOUT",
    CREATE_DATACHANNEL_ERROR = "CREATE_DATACHANNEL_ERROR",
    DATACHANNEL_CONNECTION_TIMEOUT = "DATACHANNEL_CONNECTION_TIMEOUT",
    PROHIBITED_OPERATION = "PROHIBITED_OPERATION",
    /** 鉴黄上传失败 */
    IMAGE_MODERATION_UPLOAD_FAILED = "IMAGE_MODERATION_UPLOAD_FAILED",
    /** p2p datastream传输消息失败 */
    P2P_MESSAGE_FAILED = "P2P_MESSAGE_FAILED"
}

export declare class AgoraSpecStatsFilter extends AgoraWebRTCStatsFilter {
    private _stats;
    private report;
    /**
     * spec stats 没有发送/接受帧率统计，手动统计
     */
    /** 统计帧解码间隔 **/
    private lastDecodeVideoReceiverStats;
    private lastVideoFramesRecv;
    private lastVideoFramesSent;
    private lastVideoFramesDecode;
    private lastVideoFramesOutput;
    private lastVideoJBDelay;
    private lastAudioJBDelay;
    /**
     * spec stats 没有发送码率统计，手动统计
     * key 为 ssrc
     */
    private mediaBytesSent;
    private mediaBytesRetransmit;
    private mediaBytesTargetEncode;
    private lastDecodeAudioReceiverStats;
    /** 统计平均解码耗时，每 60 帧更新一次 */
    private lastEncoderMs;
    protected updateStats(): Promise<void>;
    getSelectedCandidatePair(): Promise<IceCandidatePair>;
    private processCandidatePairStats;
    private processCandidateStats;
    private processAudioInboundStats;
    private processVideoInboundStats;
    private processVideoOutboundStats;
    private processAudioOutboundStats;
    private findRemoteStatsId;
    private processVideoMediaSource;
    private processAudioMediaSource;
    private processVideoTrackSenderStats;
    private processVideoTrackReceiverStats;
    private processAudioTrackSenderStats;
    private processAudioTrackReceiverStats;
    private processRemoteInboundStats;
    private getCodecFromCodecStats;
    /** from https://docs.google.com/document/d/1z-D4SngG36WPiMuRvWeTMN7mWQXrf1XKZwVl3Nf1BIE/edit */
    private updateSendBitrate;
}

export declare interface AgoraVideoReceiverStats extends ReceiverStats {
    /**
     * **Note:** Firefox 无法获取这个值
     */
    codec?: string;
    targetDelayMs?: number;
    renderDelayMs?: number;
    currentDelayMs?: number;
    minPlayoutDelayMs?: number;
    maxDecodeMs?: number;
    /**
     * **Note:** Firefox 和 Safari 无法获取这个值
     */
    decodeMs?: number;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    receivedFrame?: VideoFrameState;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    decodedFrame?: VideoFrameState;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    outputFrame?: VideoFrameState;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    jitterBufferMs?: number;
    firsCount: number;
    nacksCount: number;
    plisCount: number;
    framesDecodeCount: number;
    framesDroppedCount: number;
    /**
     * 总卡顿时间(s)
     * https://www.w3.org/TR/webrtc-stats/#dom-rtcinboundrtpstreamstats-totalfreezesduration
     */
    totalFreezesDuration?: number;
    /**
     * 距离上一帧解码的时间间隔(ms)
     */
    framesDecodeInterval: number;
    /**
     * 总解码卡顿时长（解码间隔 > 500ms）
     */
    framesDecodeFreezeTime: number;
    decodeFrameRate: number;
    /**
     * 实际送入渲染的帧率，不一定等于解码帧率
     */
    outputFrameRate: number;
    /**
     * **Note:** Firefox/LegacyChrome 无法获取
     */
    framesReceivedCount?: number;
    /**
     * **Note:** 这个值为 Firefox 专有
     */
    framesRateFirefox?: number;
    bytes: number;
    packets: number;
    packetsLost: number;
    packetLostRate: number;
    ssrc: number;
    qpSumPerFrame: number;
    totalInterFrameDelay?: number;
    totalSquaredInterFrameDelay?: number;
    /**
     * https://www.w3.org/TR/webrtc-stats/#dom-rtcinboundrtpstreamstats-packetsdiscarded
     */
    packetsDiscarded?: number;
    framesAssembledFromMultiplePackets?: number;
    totalProcessingDelay?: number;
    avgDecodeMs?: number;
    avgFramesAssembledFromMultiplePacketsMs?: number;
    avgProcessingDelayMs?: number;
    avgInterFrameDelayMs?: number;
    totalAssemblyTime?: number;
    keyFramesDecoded?: number;
}

export declare interface AgoraVideoSenderStats extends SenderStats {
    /**
     * **Note:** Firefox 无法获取这个值
     */
    codec?: string;
    /**
     * **Note:** Firefox/Safari 无法获取这个值
     */
    avgEncodeMs?: number;
    /** "none" / "cpu" / "bandwidth" / "other" */
    adaptionChangeReason?: "none" | "cpu" | "bandwidth" | "other";
    /**
     * **Note:** Firefox/Safari 无法获取这个值
     */
    inputFrame?: VideoFrameState;
    /**
     * **Note:** Firefox 无法获取这个值
     */
    sentFrame?: VideoFrameState;
    firsCount: number;
    nacksCount: number;
    plisCount: number;
    frameCount: number;
    bytes: number;
    packets: number;
    packetsLost: number;
    packetLostRate: number;
    ssrc: number;
    rttMs: number;
    jitterMs: number;
    qpSumPerFrame: number;
    timestamp?: DOMHighResTimeStamp;
    /** 描述编解码器（主要是视频编解码器）支持的可扩展性模式 */
    scalabilityMode?: ScalabilityMode;
    /**
     * only for video
     * Huge frames, by definition, are frames that have an encoded size at least 2.5 times the average size of the frames.
     */
    hugeFramesSent?: number;
    keyFramesEncoded?: number;
}

export declare abstract class AgoraWebRTCStatsFilter {
    onFirstVideoReceived?: (ssrcId: number) => void;
    onFirstVideoDecoded?: (ssrcId: number, width: number, height: number) => void;
    onFirstAudioReceived?: (ssrcId: number) => void;
    onFirstVideoDecodedTimeout?: (ssrcId: number) => void;
    /**
     * Firefox 目前不会触发此回调
     */
    onFirstAudioDecoded?: (ssrcId: number) => void;
    onSelectedLocalCandidateChanged?: (cur: CandidateStats, prev: CandidateStats) => void;
    onSelectedRemoteCandidateChanged?: (cur: CandidateStats, prev: CandidateStats) => void;
    /**
     * Flag used to tell if we can use the stats info. to determine freeze case.
     */
    videoIsReady: boolean;
    videoIsReady2: Record<number, boolean>;
    protected pc?: RTCPeerConnection;
    protected options: StatsFilterOptions;
    protected intervalTimer: number;
    protected stats: AgoraPCStats;
    protected isFirstVideoReceived: Record<number, boolean>;
    protected isFirstVideoDecoded: Record<number, boolean>;
    protected isFirstAudioReceived: Record<number, boolean>;
    protected isFirstAudioDecoded: Record<number, boolean>;
    protected isFirstVideoDecodedTimeout: Record<number, boolean>;
    private lossRateWindowStats;
    constructor(pc: RTCPeerConnection, options: StatsFilterOptions);
    getStats(): AgoraPCStats;
    getSelectedCandidatePair(): Promise<IceCandidatePair>;
    setVideoIsReady(isReady: boolean): void;
    setVideoIsReady2(ssrcId: number, isReady: boolean): void;
    getVideoIsReady(ssrcId: number): boolean;
    setIsFirstAudioDecoded(isFirstAudioDecoded: boolean): void;
    destroy(): void;
    /** 由上层统一计算丢包率 */
    protected calcLossRate(stats: AgoraPCStats): void;
    protected abstract updateStats(): Promise<void>;
}

export declare const appendBuffer: (buffer1: Uint8Array, buffer2: Uint8Array) => Uint8Array<ArrayBuffer>;

export declare function atom(className: string, mutexPropertyKey: string): <T extends Record<string, any>>(target: T, propertyKey: string, descriptor: TypedPropertyDescriptor<any>) => TypedPropertyDescriptor<any>;

/**
 * 观众延时级别。仅在用户角色为 `"audience"` 时生效。
 * - `1`: 低延时。
 * - `2`: （默认）超低延时。
 * @public
 */
/** @en
 *
 * The latency level of an audience member in a live interactive streaming. Takes effect only when the user role is `"audience"`.
 * - `1`: Low latency.
 * - `2`: (Default) Ultra low latency.
 */
export declare enum AudienceLatencyLevelType {
    /**
     * 低延时。
     */
    /** @en
     * Low latency.
     */
    AUDIENCE_LEVEL_LOW_LATENCY = 1,// 只对audiene有效，默认将听众的延迟（音频和视频）扩展到音视频延迟 1.2s
    /**
     * 超低延时。
     */
    /** @en
     * Ultra-low latency.
     */
    AUDIENCE_LEVEL_ULTRA_LOW_LATENCY = 2,// 只对audience有效，默认将听众的延迟（音频和视频）扩展到音视频延迟 500ms
    /**
     * @ignore
     */
    /** @en
     * @ignore
     */
    AUDIENCE_LEVEL_SYNC_LATENCY = 3
}

export declare enum AudioCodec {
    opus = "opus",
    pcma = "pcma",
    pcmu = "pcmu",
    g722 = "g722"
}

export declare function base64ToUint8Array(base64Str: string): Uint8Array;

export declare interface BrowserInfo {
    name: BrowserName | string;
    os: BrowserOS | string | null;
    version: string;
    browserVersion?: string;
    osVersion?: string;
    /**
     * Possible type:
     * console, mobile, tablet, smarttv, wearable, embedded
     * currently only focus on mobile
     * link: https://github.com/duskload/react-device-detect/blob/master/src/lib/types.js
     */
    deviceType?: string;
}

export declare enum BrowserName {
    CHROME = "Chrome",
    SAFARI = "Safari",
    EDGE = "Edge",
    FIREFOX = "Firefox",
    OPERA = "OPR",
    QQ = "QQBrowser",
    WECHAT = "MicroMessenger"
}

export declare enum BrowserOS {
    WIN_10 = "Windows 10",
    WIN_81 = "Windows 8.1",
    WIN_8 = "Windows 8",
    WIN_7 = "Windows 7",
    WIN_VISTA = "Windows Vista",
    WIN_SERVER_2003 = "Windows Server 2003",
    WIN_XP = "Windows XP",
    WIN_2000 = "Windows 2000",
    ANDROID = "Android",
    HARMONY_OS = "HarmonyOS",
    OPEN_BSD = "Open BSD",
    SUN_OS = "Sun OS",
    LINUX = "Linux",
    IOS = "iOS",
    MAC_OS = "Mac OS",
    CHROMIUM_OS = "Chromium OS",
    QNX = "QNX",
    UNIX = "UNIX",
    BEOS = "BeOS",
    OS_2 = "OS/2",
    SEARCH_BOT = "Search Bot"
}

export declare function bufferToStr(buffer: ArrayBuffer, base: number): string;

/**
 * Agora Web SDK 的编译信息。
 * @public
 */
export declare const BUILD: string;

/**
 * 因为 Typescript 的 bug，导致无法用 class 继承 Promise
 * 这里只能写脏点了
 * https://github.com/Microsoft/TypeScript/issues/15202
 */
export declare interface CancelablePromise<T> extends Promise<T> {
    cancel: () => void;
}

export declare interface CandidateStats {
    /**
     * Safari lack of `address` field.
     */
    address: string;
    candidateType: RTCIceCandidateType | "unknown";
    id: string;
    port: number;
    priority: number;
    protocol: string;
    type: string;
    relayProtocol?: string;
}

/**
 * 比较 2 个 Object 是否相等
 * **Note:** 不会递归比较，只会比较一层
 */
export declare function checkIsEqual(a: any, b: any): boolean;

export declare function checkValidArray(array: any, name: string): void;

/**
 * 所有的 check function，如果 check 失败，直接 throw error
 */
export declare function checkValidBoolean(value: any, name: string): void;

export declare function checkValidConstrainLong(num: any, name: string): void;

export declare function checkValidEnum<T>(value: any, name: string, list: T[]): void;

export declare function checkValidNumber(num: any, name: string, min?: number, max?: number, forceInteger?: boolean): void;

export declare function checkValidString(string: any, name: string, lenMin?: number, lenMax?: number, ascii?: boolean): void;

/**
 * @ignore
 */
/** @en
 * @ignore
 */
export declare type CheckVisibleResult = VisibleResultInner | VisibleHiddenResult;

declare type CheckVisibleResultInner = VisibleResultInner | VisibleHiddenResultInner;

/**
 * 用于控制客户端行为的接口。
 *
 * 在调用 {@link createClient} 创建客户端对象时，你需要配置该接口。
 *
 * > [mode]{@link ClientConfig.mode} 和 [codec]{@link ClientConfig.codec} 属性必须设置。
 * @public
 */
/** @en
 * Interface for defining the behavior of a web client.
 *
 * You need to configure it when calling the {@link createClient} method to create a web client.
 *
 * > The [mode]{@link ClientConfig.mode} and [codec]{@link ClientConfig.codec} properties are required.
 */
export declare interface ClientConfig {
    /**
     * 浏览器使用的编码格式，有以下选择：
     * - `"vp8"`: 浏览器使用 VP8 编码。
     * - `"h264"`: 浏览器使用 H.264 编码。
     * - `"vp9"`: (Beta) 浏览器使用 VP9 编码。
     * - `"av1"`:  浏览器使用 AV1 编码。
     *
     * > Safari 12.1 及之前版本不支持 VP8 编码。
     */
    /** @en
     * The codec that the Web browser uses for encoding.
     * - `"vp8"`: Use VP8 for encoding.
     * - `"h264"`: Use H.264 for encoding.
     * - `"vp9"`: (Beta) Use VP9 for encoding.
     * - `"av1"`: Use AV1 for encoding.
     *
     * > Safari 12.1 or earlier does not support the VP8 codec.
     */
    codec: SDK_CODEC;
    /**
     * @ignore
     */
    /** @en
     * @ignore
     */
    audioCodec?: SDK_AUDIO_CODEC;
    /**
     * 频道场景。
     *
     * Agora Web SDK 需知道 app 的使用场景（例如通信场景或直播场景），从而使用不同的优化手段。
     *
     * 声网频道支持以下场景:
     * - `"live"`: 直播场景，有主播和观众两种用户角色，可以通过 [setClientRole]{@link IAgoraRTCClient.setClientRole} 方法设置用户角色为主播或观众。主播可以发布和订阅音视频轨道，而观众只能订阅音视频轨道，无法发布。
     * - `"rtc"`: 通信场景，用于常见的一对一通话或群聊，频道中的任何用户可以自由说话。
     */
    /** @en
     * The channel profile.
     *
     * The SDK differentiates channel profiles and applies different optimization algorithms accordingly. For example, it prioritizes smoothness and low latency for a video call, and prioritizes video quality for a video streaming.
     *
     * The SDK supports the following channel profiles:
     * - `"live"`: Sets the channel profile as live streaming. You need to go on to call [setClientRole]{@link IAgoraRTCClient.setClientRole} to set the client as either a host or an audience. A host can send and receive audio or video, while an audience can only receive audio or video.
     * - `"rtc"`: Sets the channel profile as communication. It is used for a one-on-one call or a group call where all users in the channel can converse freely.
     */
    mode: SDK_MODE;
    /**
     * 直播场景中（[mode]{@link ClientConfig.mode} 为 `"live"` 时）的用户角色。
     *
     * 用户角色确定用户在 SDK 层的权限，包含是否可以发布和订阅音视频轨道、是否可以推流到 CDN。用户角色有 `"host"`（主播）和 `"audience"`（观众）。主播既可发布轨道，也可订阅轨道；观众不能进行 {@link publish} 操作。直播场景中的用户角色默认为观众。如需发布音视频，必须将角色角色为主播。
     *
     * 在创建客户端之后，你可以随时调用 {@link setClientRole} 来改变用户角色。
     */
    /** @en
     * The user role in a live interactive streaming (when [mode]{@link ClientConfig.mode} is `"live"`).
     *
     * The user role determines the permissions that the SDK grants to a user, such as permission to publish local streams, subscribe to remote streams, and push streams to a CDN address. You can set the user role as `"host"` or `"audience"`. A host can publish and subscribe to tracks, while an audience member can only subscribe to tracks. The default role in a live streaming is `"audience"`. Before publishing tracks, you must set the user role as `"host"`.
     *
     * After creating a client, you can call {@link setClientRole} to switch the user role.
     */
    role?: ClientRole;
    /**
     * 用户角色的具体设置，包含用户级别。
     *
     * 用户级别确定用户在其角色权限范围内可以操作和享受到的服务级别。例如对于观众，选择接收低延时还是超低延时的视频流。不同的级别会影响计费。
     */
    /** @en
     * The detailed options of the user role, including user level.
     *
     * The user level determines the level of services that a user can enjoy within the permissions of the user's role. For example, an audience can choose to receive remote streams with low latency or ultra low latency. Levels affect prices.
     */
    clientRoleOptions?: ClientRoleOptions;
    /**
     * @ignore
     * HTTP 代理服务器域名。
     *
     * Agora Web SDK 还提供 [startProxyServer]{@link IAgoraRTCClient.startProxyServer} 方法支持云代理服务，详见使用云代理。
     */
    /** @en
     * @ignore
     */
    proxyServer?: string;
    /**
     * @ignore
     * TURN 服务器设置。
     *
     * Agora Web SDK 还提供 [startProxyServer]{@link IAgoraRTCClient.startProxyServer} 方法支持云代理服务，详见使用云代理。
     */
    /** @en
     * @ignore
     */
    turnServer?: TurnServerConfig;
    /**
     * @ignore
     * SDK 内 HTTP/HTTPS 请求的重试策略。
     *
     * 关于重试策略的详细信息请参考 [RetryConfiguration]{@link RetryConfiguration}。
     */
    /** @en
     * @ignore
     */
    httpRetryConfig?: RetryConfiguration;
    /**
     * @ignore
     * SDK 内 WebSocket 连接的重试策略。
     *
     * 关于重试策略的详细信息请参考 [RetryConfiguration]{@link RetryConfiguration}。
     */
    /** @en
     * @ignore
     */
    websocketRetryConfig?: RetryConfiguration;
    /**
     * @internal
     * 暂时不开放给用户配置
     * 如果请求网关时网关断开，打开这个配置会等待网关恢复而不是直接抛错，默认开启
     */
    forceWaitGatewayResponse?: boolean;
}

export declare enum ClientEvents {
    PEERCONNECTION_STATE_CHANGE = "peerconnection-state-change",
    CONNECTION_STATE_CHANGE = "connection-state-change",
    MEDIA_RECONNECT_START = "media-reconnect-start",
    MEDIA_RECONNECT_END = "media-reconnect-end",
    IS_USING_CLOUD_PROXY = "is-using-cloud-proxy",
    USER_JOINED = "user-joined",
    USER_LEAVED = "user-left",
    USER_PUBLISHED = "user-published",
    USER_UNPUBLISHED = "user-unpublished",
    USER_INFO_UPDATED = "user-info-updated",
    CLIENT_BANNED = "client-banned",
    CHANNEL_MEDIA_RELAY_STATE = "channel-media-relay-state",
    CHANNEL_MEDIA_RELAY_EVENT = "channel-media-relay-event",
    VOLUME_INDICATOR = "volume-indicator",
    CRYPT_ERROR = "crypt-error",
    ON_TOKEN_PRIVILEGE_WILL_EXPIRE = "token-privilege-will-expire",
    ON_TOKEN_PRIVILEGE_DID_EXPIRE = "token-privilege-did-expire",
    NETWORK_QUALITY = "network-quality",
    STREAM_TYPE_CHANGED = "stream-type-changed",
    STREAM_FALLBACK = "stream-fallback",
    RECEIVE_METADATA = "receive-metadata",
    STREAM_MESSAGE = "stream-message",
    LIVE_STREAMING_ERROR = "live-streaming-error",
    LIVE_STREAMING_WARNING = "live-streaming-warning",
    EXCEPTION = "exception",
    ERROR = "error",
    P2P_LOST = "p2p_lost",
    JOIN_FALLBACK_TO_PROXY = "join-fallback-to-proxy",
    CHANNEL_FALLBACK_TO_WEBSOCKET = "channel-fallback-to-websocket",
    MEDIA_CONNECTION_TYPE_CHANGE = "media-connection-type-change",
    PUBLISHED_USER_LIST = "published-user-list",
    /** @internal */
    CONTENT_INSPECT_CONNECTION_STATE_CHANGE = "content-inspect-connection-state-change",
    /** @internal */
    CONTENT_INSPECT_ERROR = "content-inspect-error",
    /** @internal */
    CONTENT_INSPECT_RESULT = "content-inspect-result",
    IMAGE_MODERATION_CONNECTION_STATE_CHANGE = "image-moderation-connection-state-change"
}

/**
 * 用户角色。
 * - `"host"`: 直播场景中的主播，可以发布和订阅音视频轨道。
 * - `"audience"`: 直播场景中的观众，只能订阅，不能发布音视频轨道。
 * @public
 */
/** @en
 * The user role in a live broadcast channel.
 * - `"host"`: Host. A host can both publish tracks and subscribe to tracks.
 * - `"audience"`: Audience. An audience can only subscribe to tracks.
 */
export declare type ClientRole = "audience" | "host";

/**
 * 用户角色具体设置，包含用户级别。
 *
 * 用于 {@link ClientConfig.clientRoleOptions} 或 [AgoraRTCClient.setClientRole]{@link IAgoraRTCClient.setClientRole} 方法。
 * @public
 */
/** @en
 * The detailed options of the user role, including the user level.
 *
 * Used by the {@link ClientConfig.clientRoleOptions} property or the [AgoraRTCClient.setClientRole]{@link IAgoraRTCClient.setClientRole} method.
 */
export declare interface ClientRoleOptions {
    /**
     * 直播场景中的观众延时级别。
     *
     * > 注意事项：
     * > - 仅在用户角色设为 `"audience"` 时生效。
     * > - 不同的级别会影响计费。
     */
    /** @en
     * The latency level of an audience member in a live interactive streaming.
     *
     * > Note:
     * > - Takes effect only when the user role is `"audience"`.
     * > - Levels affect prices.
     */
    level: AudienceLatencyLevelType;
    /**
     * @ignore
     * 设置角色实际延迟, 范围0-3000ms
     */
    /** @en
     * @ignore
     */
    delay?: number;
}

export declare type CloudProxyServerMode = "disabled" | "proxy3" | "proxy4" | "proxy5" | "proxy6" | "fallback";

/**
 * 判断数组内的元素是否相同（不要求顺序相同）
 */
export declare function compareArray<T>(a: T[], b: T[]): boolean;

export declare function concurrent<T = void>(uuid: string, concurrency: number, method: Function, ...params: any[]): Promise<T>;

/**
 * 连接断开的原因。
 */
/** @en
 * Reason for the disconnection.
 */
export declare enum ConnectionDisconnectedReason {
    /** 用户正常退出。 */
    /** @en
     * The user has left the channel.
     */
    LEAVE = "LEAVE",
    /** 网络异常，经过重试后不可恢复。 */
    /** @en
     * The network is down, and cannot recover after retry.
     */
    NETWORK_ERROR = "NETWORK_ERROR",
    /** 服务端返回出现异常，通常是因为集成过程中参数有误。 */
    /** @en
     * The server returns an error. This is usually caused by incorrect parameter settings.
     */
    SERVER_ERROR = "SERVER_ERROR",
    /** 当前用户被踢出。 */
    /** @en
     * The user is banned.
     */
    UID_BANNED = "UID_BANNED",
    /**
     * @ignore
     */
    /** @en
     * @ignore
     */
    FALLBACK = "FALLBACK",
    /** 当前 IP 被踢出。 */
    /** @en
     * The IP is banned.
     */
    IP_BANNED = "IP_BANNED",
    /** 当前频道被禁用。 */
    /** @en
     * The channel is banned.
     */
    CHANNEL_BANNED = "CHANNEL_BANNED",
    /**
     * @ignore
     * license字段无上报
     */
    /** @en
     * @ignore
     */
    LICENSE_MISSING = "LICENSE_MISSING",
    /**
     * @ignore
     * license过期仍然登录
     */
    /** @en
     * @ignore
     */
    LICENSE_EXPIRED = "LICENSE_EXPIRED",
    /**
     * @ignore
     * license使用分钟数超过限制
     */
    /** @en
     * @ignore
     */
    LICENSE_MINUTES_EXCEEDED = "LICENSE_MINUTES_EXCEEDED",
    /**
     * @ignore
     * license使用时间段不合法
     */
    /** @en
     * @ignore
     */
    LICENSE_PERIOD_INVALID = "LICENSE_PERIOD_INVALID",
    /**
     * @ignore
     * 同一时间相同license存在不同设备
     */
    /** @en
     * @ignore
     */
    LICENSE_MULTIPLE_SDK_SERVICE = "LICENSE_MULTIPLE_SDK_SERVICE",
    /**
     * @ignore
     * license不合法
     */
    /** @en
     * @ignore
     */
    LICENSE_ILLEGAL = "LICENSE_ILLEGAL",
    /**
     * 当前用户的 Token 已过期。
     */
    /** @en
     * The user's token expires.
     */
    TOKEN_EXPIRE = "TOKEN_EXPIRE"
}

/**
 * SDK 和声网服务器的连接状态，可以通过 [connectionState]{@link IAgoraRTCClient.connectionState} 获取。
 *
 * SDK 和服务器有以下 5 种连接状态:
 * - `"DISCONNECTED"`: 连接断开。该状态表示用户处于以下任一阶段：
 *   - 尚未通过 [join]{@link IAgoraRTCClient.join} 加入频道。
 *   - 已经通过 [leave]{@link IAgoraRTCClient.leave} 离开频道。
 *   - 被踢出频道或者连接失败等异常情况。
 * - `"CONNECTING"`: 正在连接中。当调用 [join]{@link IAgoraRTCClient.join} 时为此状态。
 * - `"CONNECTED"`: 已连接。该状态表示用户已经加入频道，可以在频道内发布或订阅媒体流。
 * - `"RECONNECTING"`: 正在重连中。因网络断开或切换而导致 SDK 与服务器的连接中断，SDK 会自动重连，此时连接状态变为 `"RECONNECTING"`。
 * - `"DISCONNECTING"`: 正在断开连接。在调用 [leave]{@link IAgoraRTCClient.leave} 的时候为此状态。
 * @public
 */
/** @en
 * Connection state between the SDK and Agora's edge server.
 *
 * You can get the connection state through [connectionState]{@link IAgoraRTCClient.connectionState}.
 *
 * The connection between the SDK and the edge server has the following states:
 * - `"DISCONNECTED"`: The SDK is disconnected from the server.
 *  - This is the initial state until you call [join]{@link IAgoraRTCClient.join}.
 *  - The SDK also enters this state after you call [leave]{@link IAgoraRTCClient.leave}, when the user is banned, or when the connection fails.
 * - `"CONNECTING"`: The SDK is connecting to the server. The SDK enters this state when you call [join]{@link IAgoraRTCClient.join}.
 * - `"CONNECTED"`: The SDK is connected to the server and joins a channel. The user can now publish streams or subscribe to streams in the channel.
 * - `"RECONNECTING"`: The SDK is reconnecting to the server. If the connection is lost because the network is down or switched, the SDK enters this state.
 * - `"DISCONNECTING"`: The SDK is disconnecting from the server. The SDK enters this state when you call [leave]{@link IAgoraRTCClient.leave}.
 */
export declare type ConnectionState = "DISCONNECTED" | "CONNECTING" | "RECONNECTING" | "CONNECTED" | "DISCONNECTING";

export declare function constrainLongToNumber(value: ConstrainULong): number;

/**
 * 将输入字符串转换为指定长度的Uint8Array。
 * 如果字符串不足指定长度则填充0，超出指定长度则截断。
 *
 * @param input - 要转换的字符串
 * @param length - 固定长度，默认为16
 * @returns 指定长度的Uint8Array
 */
export declare function convertStringToFixedLengthUint8Array(input: string, length?: number): Uint8Array;

export declare function createDefer<T = void>(): Deferred<T>;

/**
 * @internal
 * 快速创建一个内部使用的 unexpected error 对象
 */
export declare function createInternalUnexpectedError(message: string, data?: any): AgoraRTCError;

export declare function createResolvedDefer<T = void>(value: T): Deferred<T>;

export declare function createTimeoutDefer(duration?: number): Deferred<void>;

export declare function createWebRTCStatsFilter(pc: RTCPeerConnection, updateInterval?: number, lossRateInterval?: number, freezeRateLimit?: number, firstVideoDecodedTimeout?: number): AgoraWebRTCStatsFilter;

export declare const CRYPTO_HEADER_LENGTH = 10;

export declare const CRYPTO_ITERATIONS = 1000;

export declare const CRYPTO_IV_LENGTH: number;

export declare const CRYPTO_TAG_LENGTH: number;

declare type CryptoMode = "aes-128-gcm2" | "aes-256-gcm2";

export declare interface DataChannelMessage {
    uid: number;
    stream_id: number;
    ordered: boolean;
    max_retrans_times: number;
    metadata: string;
}

export declare function decryptAesGcm(iv: Uint8Array, key: CryptoKey, payload: Uint8Array): Promise<Uint8Array>;

export declare const DEFAULT_AREAS: string[];

export declare const DEFAULT_AUDIO_RECV_STATS: AgoraAudioReceiverStats;

export declare const DEFAULT_AUDIO_SEND_STATS: AgoraAudioSenderStats;

export declare const DEFAULT_CANDIDATE_STATS: CandidateStats;

export declare const DEFAULT_PC_STATS: AgoraPCStats;

export declare const DEFAULT_RETRY_CONFIG: RetryConfiguration;

export declare const DEFAULT_TURN_CONFIG: TurnServerConfig;

export declare const DEFAULT_VIDEO_RECV_STATS: AgoraVideoReceiverStats;

export declare const DEFAULT_VIDEO_SEND_STATS: AgoraVideoSenderStats;

export declare interface Deferred<T = void> {
    promise: Promise<T>;
    isResolved: boolean;
    isRejected: boolean;
    isFinished: boolean;
    value?: T;
    resolve: (value: T) => void;
    reject: (reason: unknown) => void;
    cancel: (reason?: unknown) => void;
}

/** Deprecated */
export declare function deprecatedGetBrowserInfo(userAgent?: string): BrowserInfo;

export declare function detectSecureContext(): boolean;

export declare function dividePackage<T>(source: T, key: string, mtu: number): T[];

export declare function domLoadedPromise(): Promise<void>;

declare type Electron = any;

/**
 * 通过 {@link getElectronScreenSources} 获取的 Electron 下屏幕共享源信息。
 *
 * 详见 [Electron 官方文档](https://www.electronjs.org/docs/api/structures/desktop-capturer-source)。
 * @public
 */
/** @en
 * Information of the sharing screen source on Electron, which is retrieved by calling {@link getElectronScreenSources}.
 *
 * See [DesktopCapturerSource](https://www.electronjs.org/docs/api/structures/desktop-capturer-source) in the Electron API documentation for details.
 */
export declare interface ElectronDesktopCapturerSource {
    /**
     * Electron 屏幕共享源的 ID。
     */
    /** @en
     * The ID of the screen source.
     */
    id: string;
    /**
     * Electron 屏幕共享源的名称。
     */
    /** @en
     * The name of the screen source.
     */
    name: string;
    /**
     * Electron 屏幕共享源的快照。
     *
     * 详见 [ElectronNativeImage](http://electron.atom.io/docs/api/native-image)。
     */
    /** @en
     * The thumbnail of the screen source.
     *
     * See [ElectronNativeImage](https://electronjs.org/docs/api/native-image#nativeimage) for details.
     */
    thumbnail: IElectronNativeImage;
}

declare class ElementVisibleChecker {
    private _clientSize;
    private getClientWidth;
    private getClientHeight;
    private getStyle;
    private checkCssVisibleProperty;
    private checkPropertyUpToAllParentNodes;
    private checkActualCssVisibleIncludeInherit;
    private getSizeAboutClient;
    private checkActualSize;
    private elementFromPoint;
    private checkCoverForAPoint;
    private getPointPositionList;
    private checkElementCover;
    private checkSizeIsVisible;
    private checkSizeOfPartInClient;
    private returnHiddenResult;
    checkOneElementVisible: (element?: HTMLElement) => CheckVisibleResultInner;
    private checkElementIsMountedOnDom;
}

export declare const elementVisibleChecker: ElementVisibleChecker;

/**
 * emit 一个事件，并假定 handler 会同步地处理事件，将 handler 的返回作为这个函数的返回，将 handler 的异常直接抛出
 */
export declare function emitAsInvoker<T = any, E = AgoraRTCError>(emitter: EventEmitter, event: string, ...args: any[]): T;

/**
 * emit 一个事件，并假定 handler 会同步地处理事件，将 handler 的返回作为这个函数的返回，将 handler 的异常直接抛出
 * **Note:**
 * 如果没有 handler 直接返回 `null`
 */
export declare function emitAsInvokerNoResponse<T = any, E = AgoraRTCError>(emitter: EventEmitter, event: string, ...args: any[]): T | null;

/**
 * emit 一个事件，并假定 handler 会异步地处理事件，将 handler 的返回作为这个函数的 Promise 的返回，将 handler 的异常作为 Promise 的异常
 * 如果检测到没有 handler 会抛出 UNEXPECTED_ERROR
 */
export declare function emitAsPromise<T = any>(emitter: EventEmitter, event: string, ...args: any[]): Promise<T>;

/**
 * emit 一个事件，并假定 handler 会异步地处理事件，将 handler 的返回作为这个函数的 Promise 的返回，将 handler 的异常作为 Promise 的异常
 * 要求返回的 Promise 为 Promise<void>
 * 如果检测到没有 handler 会直接 resolve
 */
export declare function emitAsPromiseNoResponse(emitter: EventEmitter, event: string, ...args: any[]): Promise<void>;

export declare function encryptAesGcm(iv: Uint8Array, key: CryptoKey, payload: Uint8Array): Promise<Uint8Array>;

/**
 * 加密方案，在调用 {@link setEncryptionConfig} 时使用。包含以下几种：
 * - `"aes-128-xts"`: 128 位 AES 加密，XTS 模式。
 * - `"aes-256-xts"`: 256 位 AES 加密，XTS 模式。
 * - `"aes-128-gcm"`: 128 位 AES 加密，GCM 模式。
 * - `"aes-256-gcm"`: 256 位 AES 加密，GCM 模式。
 * - `"aes-128-gcm2"`: 128 位 AES 加密，GCM 模式，加盐。
 * - `"aes-256-gcm2"`: 256 位 AES 加密，GCM 模式，加盐。
 * - `"aes-128-ecb"`: 128 位 AES 加密，ECB 模式。
 * - `"sm4-128-ecb"`: 128 位 SM4 加密，ECB 模式。
 * - `"none"`: 不加密。
 * @public
 */
/** @en
 * The encryption mode, which is used in the {@link setEncryptionConfig} method call.
 * - `"aes-128-xts"`: 128-bit AES encryption, XTS mode.
 * - `"aes-256-xts"`: 256-bit AES encryption, XTS mode.
 * - `"aes-128-gcm"`: 128-bit AES encryption, GCM mode.
 * - `"aes-256-gcm"`: 256-bit AES encryption, GCM mode.
 * - `"aes-128-gcm2"`: 128-bit AES encryption, GCM mode, with salt.
 * - `"aes-256-gcm2"`: 256-bit AES encryption, GCM mode, with salt.
 * - `"aes-128-ecb"`: 128-bit AES encryption, ECB mode.
 * - `"sm4-128-ecb"`: 128-bit SM4 encryption, ECB mode.
 * - `"none"`: No encryption.
 */
export declare type EncryptionMode = "aes-128-xts" | "aes-256-xts" | "aes-128-ecb" | "sm4-128-ecb" | "aes-128-gcm" | "aes-256-gcm" | "aes-128-gcm2" | "aes-256-gcm2" | "none";

export declare const encryptRSA: (secret: string) => Promise<string>;

/**
 * `EventEmitter` 类提供了定义、触发和处理事件的方式。
 */
/** @en
 * The `EventEmitter` class provides a way to define, emit, and handle events.
 */
export declare class EventEmitter {
    private _events;
    /**
     * 指定一个事件名，获取当前所有监听这个事件的回调函数。
     *
     * @param event - 事件名称。
     */
    /** @en
     * Gets all the listeners for a specified event.
     *
     * @param event The event name.
     */
    getListeners(event: string): Function[];
    /**
     * 监听一个指定的事件，当事件触发时会调用传入的回调函数。
     *
     * @param event - 指定事件的名称。
     * @param listener - 传入的回调函数。
     */
    /** @en
     * Listens for a specified event.
     *
     * When the specified event happens, the SDK triggers the callback that you pass.
     * @param event The event name.
     * @param listener The callback to trigger.
     */
    on(event: string, listener: Function): void;
    /** @internal */
    addListener: (event: string, listener: Function) => void;
    /**
     * 监听一个指定的事件，当事件触发时会调用传入的回调函数。
     *
     * 当监听后事件第一次触发时，该监听和回调函数就会被立刻移除，也就是只监听一次指定事件。
     *
     * @param event - 指定事件的名称。
     * @param listener - 传入的回调函数。
     */
    /** @en
     * Listens for a specified event once.
     *
     * When the specified event happens, the SDK triggers the callback that you pass and then removes the listener.
     * @param event The event name.
     * @param listener The callback to trigger.
     */
    once(event: string, listener: Function): void;
    /**
     * 取消一个指定事件的监听。
     *
     * @param event - 指定事件的名称。
     * @param listener - 监听事件时传入的回调函数。
     */
    /** @en
     * Removes the listener for a specified event.
     *
     * @param event The event name.
     * @param listener The callback that corresponds to the event listener.
     */
    off(event: string, listener: Function): void;
    /**
     * 指定一个事件，取消其所有的监听。
     *
     * @param event - 指定事件的名称，如果没有指定事件，则取消所有事件的所有监听。
     */
    /** @en
     * Removes all listeners for a specified event.
     *
     * @param event The event name. If left empty, all listeners for all events are removed.
     */
    removeAllListeners(event?: string): void;
    /** @internal */
    emit(event: string, ...args: any[]): void;
    /** @internal */
    safeEmit(event: string, ...args: any[]): void;
    private _indexOfListener;
}

/**
 *
 * @param list
 * @param predicate
 * @returns
 * 返回满足条件的元素的索引
 */
export declare function findIndexs<T>(list: T[], predicate: (item: T) => boolean): number[];

export declare function generateIv(mode: CryptoMode, password: Uint8Array, salt: Uint8Array): Promise<Uint8Array>;

export declare function generateKey(mode: CryptoMode, password: Uint8Array, salt: Uint8Array): Promise<CryptoKey>;

export declare function generateProcessID(): string;

export declare function generateSessionID(): string;

/**
 * dataview-bigint-polyfill
 * from Chrome 67+, Firefox 68+, Safari 15+
 * to 67+ , Firefox 68+, Safari 14+
 */
export declare function getBigInt64(dataView: DataView, byteOffset: number, littleEndian: boolean | undefined): bigint;

export declare function getBigUint64(dataView: DataView, byteOffset: number, littleEndian?: boolean): bigint;

export declare function getBrowserInfo(userAgent?: string): Readonly<BrowserInfo>;

export declare function getBrowserOS(): BrowserOS | string | null;

export declare function getBrowserVersion(): string;

export declare function getChangedKeysFromObject<T extends object>(a: T, b: T): (keyof T)[];

export declare function getChromeKernelVersion(): string | null;

export declare function getElectronInstance(): Electron | null;

/**
 * 获取通过 post 接收的总字节数
 */
export declare function getHTTPRecvBytes(): number;

/**
 * 获取通过 post 发送的总字节数
 */
export declare function getHTTPSendBytes(): number;

export declare function getMessageEncoding(text: string): Uint8Array;

export declare function getMultiUnilbsFormDataByteLength(data: FormData): number;

export declare function getOSWithVersion(): string;

/**
 * 获取可变的全局变量
 * @param key - 全局变量的 key
 *
 */
export declare function getParameter(key: keyof typeof MUTABLE_PARAMS): any;

export declare function getRandomString(length: number, prefix: string): string;

export declare function getRetryWaitTime(retryCount: number, config: RetryConfiguration): number;

export declare function getUniqueList<T>(list: T[]): T[];

export declare function getUTF8StringByteLength(str: string): number;

export declare function hexToBytes(hex: string): Uint8Array;

/**
 * AgoraSDK 的错误码对象
 * @ignore
 */
/** @en
 * @ignore
 */
export declare interface IAgoraRTCError extends Error {
    readonly code: AgoraRTCErrorCode;
    readonly message: string;
    readonly data?: any;
    readonly name: string;
    toString(): string;
    print(level?: "error" | "warning", logger?: any): IAgoraRTCError;
    throw(logger?: any): never;
}

export declare interface IceCandidatePair {
    local: CandidateStats;
    remote: CandidateStats;
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
export declare interface IElectronNativeImage {
    toDataURL(): string;
    getSize(): {
        width: number;
        height: number;
    };
    resize(options: {
        width: number;
    }): IElectronNativeImage;
}

export declare const IS_GLOBAL_VERSION: boolean;

export declare function isAboveChrome(minVersion: number): boolean;

export declare function isAboveEdge(minVersion: number): boolean;

export declare function isAboveFirefox(minVersion: number): boolean;

export declare function isAboveIOS(minVersion: number): boolean;

export declare function isAboveIOS13(): boolean;

export declare function isAboveIOS15_1(): boolean;

export declare function isAboveIOS15_2(): boolean;

export declare function isAboveIOS16_0(): boolean;

export declare function isAboveOpera(minVersion: number): boolean;

export declare function isAboveSafari(minVersion: number): boolean;

export declare function isAndroid(): boolean;

export declare function isAndroidChromium(): boolean;

export declare function isBelowChrome(maxVersion: number): boolean;

export declare function isBelowIOS(maxVersion: number, subVersion?: number, isExcluded?: boolean): boolean;

export declare function isBelowIOS14_6(): boolean;

export declare function isBelowSafari(maxVersion: number, subVersion?: number, isExcluded?: boolean): boolean;

export declare function isBetweenBrowser(browser: BrowserName, minVersion: number, maxVersion?: number): boolean;

export declare function isChrome(): boolean;

export declare function isChromeBelow90(): boolean;

export declare function isChromeKernel(): boolean;

export declare function isClientConfig(config: ClientConfig): config is ClientConfig;

export declare function isClientRole(role: any): role is ClientRole;

export declare function isClientRoleOptions(options: any): options is ClientRoleOptions;

export declare function isEdge(): boolean;

export declare function isElectron(): boolean;

export declare function isEmpty(n: any): boolean;

export declare function isEncryptionMode(encryptionMode: any): encryptionMode is EncryptionMode;

export declare function isFirefox(): boolean;

export declare function isHarmonyOS(): boolean;

export declare const isHttpsEnv: () => boolean;

export declare function isInPage(node: Node): boolean;

export declare function isIOS(): boolean;

export declare function isIOS13(): boolean;

export declare function isIOS15(): boolean;

export declare function isIOS15_0(): boolean;

export declare function isIOS16(): boolean;

export declare function isIpadOS(): boolean;

export declare function isJsonEqual(obj1: any, obj2: any): boolean;

export declare function isLegacyChrome(): boolean;

export declare function isMacOS(): boolean;

export declare function isMobile(): boolean;

export declare function isMobileAndTabletType(): boolean;

export declare function isOpera(): boolean;

export declare function isP2PTransport(transport: P2PTransportType): transport is P2PTransportType;

/**
 * 判断是否是 页面录制，示例ua：
 * Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko)
 * Chrome/103.0.5060.81 Safari/537.36 AgoraWebView/2.10.14 /Epveo+QtQO0p4Q0tKOBNbahtQM=
 */
export declare function isPageRecording(): boolean;

export declare function isPromise(obj: any): boolean;

export declare function isQQBrowser(): boolean;

export declare function isRetryConfiguration(config: RetryConfiguration): config is RetryConfiguration;

export declare function isRTCIceServerList(iceServers: any): iceServers is RTCIceServer[];

export declare function isSafari(): boolean;

export declare function isSupportedPC(): boolean;

/** Known Issue:
 * - iOS Chrome(or the other WkWebView) request desktop website,
 * the OS version is MAC_OS so that we could not figure out if it support webrtc,
 * we will relax the constraints in this situation */
export declare function isSupportedWkWebview(): boolean;

export declare function isTurnServerConfig(turnServer: any): turnServer is TurnServerConfig;

export declare function isValidString(string: any, lenMin?: number, lenMax?: number, ascii?: boolean): boolean;

export declare function isWebKit(): boolean;

export declare function isWechatBrowser(): boolean;

export declare function isWindows(): boolean;

export declare function isWkWebview(): boolean;

export declare interface JoinChannelServiceRecord {
    urls: string[];
    startTs: number;
    endTs: number | undefined;
    sessionId: string;
    errors?: Error[];
    uid?: UID;
    status: "pending" | "success" | "error" | "timeout" | "aborted";
    service: "stringUID" | "chooseServer" | "gateway";
    cloudProxyMode: "disabled" | "normal" | "443only" | "proxy3" | "proxy4" | "proxy5" | "proxy6" | "fallback";
}

export declare function jsonClone<T>(obj: T): T;

export declare interface KeyMetrics {
    clientCreated?: number;
    joinStart?: number;
    joinEnd?: number;
    requestAPStart?: number;
    requestAPEnd?: number;
    joinGatewayStart?: number;
    joinGatewayEnd?: number;
    peerConnectionStart?: number;
    peerConnectionEnd?: number;
    descriptionStart?: number;
    iceConnectionEnd?: number;
    datachannelOpen?: number;
    publish: {
        trackId: string;
        type: "video" | "audio";
        publishStart?: number;
        publishEnd?: number;
    }[];
    subscribe: {
        userId: UID;
        type: "video" | "audio";
        subscribeStart?: number;
        subscribeEnd?: number;
        firstFrame?: number;
        streamAdded?: number;
        firstDecoded?: number;
    }[];
}

/**
 * 混合云/私有化平台的配置参数。用于 {@link setLocalAccessPointsV2} 方法。
 *
 * - `accessPoints`：访问节点，包含以下属性：
 *   - `serverList`：IP 地址列表或 hostname（主机名，用于指定 SDK 访问的 Agora SD-RTN™ 边缘节点）列表。
 *   - `domain`：域名。如果传入的 `serverList` 为 IP 地址列表，SDK 会拼接传入的 IP 地址和 `domain` 生成 hostname 列表。此时 `domain` 用于指定 SDK 接入 Agora SD-RTN™ 的网关。
 *   - `port`：（可选）端口号。如果传入，端口号会和 hostname 拼在一起。
 * - `log`：（可选）日志上传服务。包含可选属性 `hostname` 和 `port`。如果要设置日志上传服务，请确保你已经调用 {@link enableLogUpload} 开启 SDK 的日志上传功能。
 * - `report`：（可选）事件上报服务。默认关闭。包含可选属性 `hostname` 和 `port`。
 * - `cds`：预留参数，暂不支持。
 *
 * 可选属性如果不填，将使用对应的默认值：
 *
 *   |属性|默认值|
 *   |:----:|:----:|
 *   |`log.hostname`| `accessPoints` 所包含的 hostname 列表中的第一个|
 *   |`log.port`|6444|
 *   |`report.hostname`| `accessPoints` 所包含的 hostname 列表中的前两个|
 *   |`report.port`|6443|
 *   |`accessPoints.port`|443|
 *
 *
 * ```javascript
 * client.setLocalAccessPointsV2({
 *   accessPoints: {
 *     serverList:["192.168.1.1","192.168.2.2"],
 *     domain: 'test.agora.io'
 *   },
 *   // 必须先调用 enableLogUpload，下面的日志服务设置才生效
 *   log:{
 *     hostname:["abc.com"],
 *     port: 3000
 *   },
 *   // 开启事件上报服务并配置参数
 *   report:{
 *     hostname:["example.com"],
 *     port: 8080
 *   }
 * })
 * ```
 *
 */
/** @en
 * @ignore
 *
 * -----暂不对外的备用内容-----
 * - `cds`：（可选）配置下发服务。默认关闭。包含可选属性 `hostname` 和 `port`。
 *
 * 默认值
 *   |`cds.hostname`| `accessPoints` 所包含的 hostname 列表|
 *   |`cds.port`|443|
 */
export declare type LocalAccessPointConfig = {
    [serve in "log" | "report" | "cds"]?: {
        hostname?: string[];
        port?: number;
    };
} & {
    accessPoints: {
        serverList: string[];
        domain: string;
        port?: number;
    };
};

export declare function md5(inputString: string): string;

export declare const MUTABLE_GATEWAY_PARAMS: {
    ENABLE_PUBLISHED_USER_LIST: boolean;
    /**
     * 128/50模式最多订阅人数
     * */
    MAX_SUBSCRIPTION: number;
    /**
     * TOPN接收端设置下行选路路数
     * */
    SUBSCRIBE_AUDIO_FILTER_TOPN: any;
    /**
     * 发送端设置当前发送的音频流是否要参与选路
     * */
    ENABLE_PUBLISH_AUDIO_FILTER: any;
    /**
     * 是否开启license校验
     * */
    ENABLE_USER_LICENSE_CHECK: boolean;
    /** 上行是否使用RTX*/
    USE_PUB_RTX: boolean;
    /** 下行是否使用RTX*/
    USE_SUB_RTX: boolean;
    /**
     * 是否禁用 RSFEC（默认不禁用）
     */
    DISABLE_FEC: any;
    ENABLE_NTP_REPORT: boolean;
    ENABLE_INSTANT_VIDEO: boolean;
    /**
     * 是否开启DataStream2.0能力，default: false
     * */
    ENABLE_DATASTREAM_2: boolean;
    /**
     * 4.19.0之后支持踢人大重连错误code: 28
     */
    ENABLE_USER_AUTO_REBALANCE_CHECK: boolean;
    USE_XR: boolean;
    /** 透传服务器字段 */
    ENABLE_LOSSBASED_BWE: boolean;
    /** AutCC 开关 */
    ENABLE_AUT_CC: boolean;
    /** CC fallback: 网关透传字段 */
    ENABLE_CC_FALLBACK: any;
    /**
     * 开启前置PC，注意 ：不适合过程中修改，只在初始化时设置
     */
    ENABLE_PREALLOC_PC: boolean;
    SUBSCRIBE_TWCC: boolean;
    PUBLISH_TWCC: boolean;
    ENABLE_SVC: boolean;
    ENABLE_SVC_DEFAULT_CODECS: string[];
    /** 添加svc支持的codec来启用svc功能，默认不开 */
    SVC: any[];
    ENABLE_FULL_LINK_AV_SYNC: boolean;
    /** svc 模式 */
    SVC_MODE: any;
    /**
     * 预订阅数目
     */
    PRE_SUB_NUM: number;
    /**
     * 是否开启预订阅
     */
    ENABLE_PRE_SUB: boolean;
    ENABLE_AUT_FEEDBACK: boolean;
    SVC_EXTENDED: string[];
};

/**
 * @internal
 */
export declare const MUTABLE_PARAMS: {
    /**
     * 选用ap返回的candidate，作为媒体节点
     */
    USE_CANDIDATE_FROM_AP_DETAIL: boolean;
    /**
     * 请求 AP 时，携带拓展信息，方便 AP 选择节点
     */
    AP_REQUEST_DETAIL: any;
    /**
     * 是否开启 角色 选取边缘节点
     */
    ENABLE_ROLE_SELECT_EDGE: boolean;
    /**
     * 下发当前客户的角色
     */
    CLIENT_ROLE_OPTIONS: any;
    /**
     * NEW RESTART ICE 策略，支持配置下发
     */
    RESTART_SEQUENCE: string[];
    COMPATIBLE_SDP_EXTENSION: string[];
    /**
     * 码率控制参数，！！！ 只用于配置下发
     * 设计原因：
     * 早期SDK (4.23.0 以前)，WebSDK 只支持_web limit_bitrate名字的配置下发控制码率，极大的限制了配置下发的灵活性
     * 所以新留了一个配置下发的口子，用于控制码率
     */
    LIMIT_BITRATE: any;
    /**
     * 网关透传口子，v4.23.0 以后，sdk 无需额外新增网关透传字段
     */
    EXPERIMENTS: {};
    ENABLE_PUBLISHED_USER_LIST: boolean;
    /**
     * 128/50模式最多订阅人数
     * */
    MAX_SUBSCRIPTION: number;
    /**
     * TOPN接收端设置下行选路路数
     * */
    SUBSCRIBE_AUDIO_FILTER_TOPN: any;
    /**
     * 发送端设置当前发送的音频流是否要参与选路
     * */
    ENABLE_PUBLISH_AUDIO_FILTER: any;
    /**
     * 是否开启license校验
     * */
    ENABLE_USER_LICENSE_CHECK: boolean;
    /** 上行是否使用RTX*/
    USE_PUB_RTX: boolean;
    /** 下行是否使用RTX*/
    USE_SUB_RTX: boolean;
    /**
     * 是否禁用 RSFEC（默认不禁用）
     */
    DISABLE_FEC: any;
    ENABLE_NTP_REPORT: boolean;
    ENABLE_INSTANT_VIDEO: boolean;
    /**
     * 是否开启DataStream2.0能力，default: false
     * */
    ENABLE_DATASTREAM_2: boolean;
    /**
     * 4.19.0之后支持踢人大重连错误code: 28
     */
    ENABLE_USER_AUTO_REBALANCE_CHECK: boolean;
    USE_XR: boolean;
    /** 透传服务器字段 */
    ENABLE_LOSSBASED_BWE: boolean;
    /** AutCC 开关 */
    ENABLE_AUT_CC: boolean;
    /** CC fallback: 网关透传字段 */
    ENABLE_CC_FALLBACK: any;
    /**
     * 开启前置PC，注意 ：不适合过程中修改，只在初始化时设置
     */
    ENABLE_PREALLOC_PC: boolean;
    SUBSCRIBE_TWCC: boolean;
    PUBLISH_TWCC: boolean;
    ENABLE_SVC: boolean;
    ENABLE_SVC_DEFAULT_CODECS: string[];
    /** 添加svc支持的codec来启用svc功能，默认不开 */
    SVC: any[];
    ENABLE_FULL_LINK_AV_SYNC: boolean;
    /** svc 模式 */
    SVC_MODE: any;
    /**
     * 预订阅数目
     */
    PRE_SUB_NUM: number;
    /**
     * 是否开启预订阅
     */
    ENABLE_PRE_SUB: boolean;
    ENABLE_AUT_FEEDBACK: boolean;
    SVC_EXTENDED: string[];
    PROCESS_ID: string;
    /**
     * 是否使用AES加密
     * block join:true
     * */
    ENCRYPT_AES: boolean;
    /**
     * 区域限定area code
     * block join:true
     * */
    AREAS: string[];
    /**
     * AP域名
     * block join:true
     * */
    WEBCS_DOMAIN: string[];
    WEBCS_DOMAIN_BACKUP_LIST: string[];
    PROXY_CS: string[];
    CDS_AP: string[];
    ACCOUNT_REGISTER: string[];
    UAP_AP: string[];
    LOG_UPLOAD_SERVER: string;
    EVENT_REPORT_DOMAIN: string;
    EVENT_REPORT_BACKUP_DOMAIN: string;
    ENABLE_EVENT_REPORT: boolean;
    GATEWAY_ADDRESS: any[];
    GATEWAY_WSS_ADDRESS: string;
    LIVE_STREAMING_ADDRESS: string;
    HTTP_CONNECT_TIMEOUT: number;
    SIGNAL_REQUEST_TIMEOUT: number;
    REPORT_STATS: boolean;
    UPLOAD_LOG: boolean;
    NOT_REPORT_EVENT: any[];
    PING_PONG_TIME_OUT: number;
    WEBSOCKET_TIMEOUT_MIN: number;
    EVENT_REPORT_SEND_INTERVAL: number;
    CONFIG_DISTRIBUTE_INTERVAL: number;
    ENABLE_CONFIG_DISTRIBUTE: boolean;
    CANDIDATE_TIMEOUT: number;
    SHOW_REPORT_INVOKER_LOG: boolean;
    SHOW_REPORT_USER_INVOKER_LOG: boolean;
    JOIN_EXTEND: string;
    PUB_EXTEND: string;
    SUB_EXTEND: string;
    FORCE_TURN: boolean;
    TURN_ENABLE_TCP: boolean;
    TURN_ENABLE_UDP: boolean;
    MAX_UPLOAD_CACHE: number;
    UPLOAD_CACHE_INTERVAL: number;
    AJAX_REQUEST_CONCURRENT: number;
    REPORT_APP_SCENARIO: any;
    GATEWAY_DOMAINS: string[];
    CONNECT_GATEWAY_WITHOUT_DOMAIN: boolean;
    WORKER_DOMAIN: string;
    TURN_DOMAIN: string;
    EVENT_REPORT_RETRY: boolean;
    CHROME_FORCE_PLAN_B: boolean;
    /** 计算 WebAudio 音量的间隔(ms) Deprecated on 4.7.0 */
    AUDIO_SOURCE_VOLUME_UPDATE_INTERVAL: number;
    /** 计算平均音量的最大音频 duration (ms) Deprecated on 4.7.0 */
    AUDIO_SOURCE_AVG_VOLUME_DURATION: number;
    /** volume-indicator 回调间隔(ms) */
    AUDIO_VOLUME_INDICATION_INTERVAL: number;
    /** 音量计算中语音所在频段的权重（0 <= w <= 16） */
    VOLUME_VOICE_WEIGHT: number;
    /** AudioTrack mute 后仍获取音量 */
    GET_VOLUME_OF_MUTED_AUDIO_TRACK: boolean;
    /** 统计信息更新事件间隔 */
    STATS_UPDATE_INTERVAL: number;
    /** 非关键事件的最大缓存数 */
    NORMAL_EVENT_QUEUE_CAPACITY: number;
    /** 是否开启自定义事件上报 */
    CUSTOM_REPORT: boolean;
    /** 自定义上报数量限制 （5s） */
    CUSTOM_REPORT_LIMIT: number;
    PROXY_SERVER_TYPE2: string;
    PROXY_SERVER_TYPE3: string | string[];
    CUSTOM_PUB_ANSWER_MODIFIER: any;
    CUSTOM_SUB_ANSWER_MODIFIER: any;
    CUSTOM_PUB_OFFER_MODIFIER: any;
    CUSTOM_SUB_OFFER_MODIFIER: any;
    DSCP_TYPE: string;
    REMOVE_NEW_CODECS: boolean;
    FRAGEMENT_LENGTH: number;
    WEBSOCKET_COMPRESS: boolean;
    SIMULCAST: boolean;
    /** RemoteSDP中预先加载并创建的Transceiver数量*/
    PRELOAD_MEDIA_COUNT: number;
    /** 触发检测video是否可见的事件的时间间隔 (ms) */
    CHECK_VIDEO_VISIBLE_INTERVAL: number;
    /** 触发检测LocalTrack Stats事件的时间间隔 (ms) */
    CHECK_LOCAL_STATS_INTERVAL: number;
    /** switch localVideoTrack profiles */
    PROFILE_SWITCH_INTERVAL: number;
    /** 不支持的codec */
    UNSUPPORTED_VIDEO_CODEC: any[];
    /** 是否轮训获取设备 */
    ENUMERATE_DEVICES_INTERVAL: boolean;
    /** 轮询获取设备的间隔 */
    ENUMERATE_DEVICES_INTERVAL_TIME: number;
    /** 是否走 token 7 鉴权流程 */
    USE_NEW_TOKEN: boolean;
    /** 对于私有云/混合云是否关闭AFB */
    CLOSE_AFB_FOR_LOCAL_AP: boolean;
    /** 使用 joinMeta 时最大并发数 */
    JOIN_MAX_CONCURRENCY: number;
    /** join 方法时候是否走 自动 fallback 到 proxy 的逻辑（信令） */
    JOIN_WITH_FALLBACK_SIGNAL_PROXY: boolean;
    /** join 方法时候走 自动 fallback 到 proxy 的逻辑（媒体） */
    JOIN_WITH_FALLBACK_MEDIA_PROXY: boolean;
    /** join 方法自动 fallback 到 proxy 触发时，媒体 fallback 是否强制走 proxy */
    JOIN_WITH_FALLBACK_MEDIA_PROXY_FORCE: boolean;
    /** join 方法自动 fallback 到 443 的检测时间 */
    JOIN_GATEWAY_TRY_443PORT_DURATION: number;
    /** join 方法是否只是用 443 端口 */
    JOIN_GATEWAY_USE_443PORT_ONLY: boolean;
    /** join 方法是否使用双与域名 */
    JOIN_GATEWAY_USE_DUAL_DOMAIN: boolean;
    /** join 方法自动 fallback 到 443 后支持自定义非 443 端口 */
    JOIN_GATEWAY_FALLBACK_PORT: number;
    /** 媒体 ice Servers 列表填入网关的 turn servers */
    USE_TURN_SERVER_OF_GATEWAY: boolean;
    /**使用42e01f h264 profile-level-id*/
    H264_PROFILE_LEVEL_ID: string;
    /** 是否使用新日志打印*/
    USE_NEW_LOG: boolean;
    LOG_VERSION: number;
    MEDIA_DEVICE_CONSTRAINTS: any;
    ENCRYPT_PROXY_USERNAME_AND_PSW: boolean;
    /**是否打印SDP*/
    SDP_LOGGING: boolean;
    /**
     * 能触发 CSP 回调的 URI 的 Hostname list
     */
    CSP_DETECTED_HOSTNAME_LIST: string[];
    /** Remote Audio Track uses Web Audio */
    REMOTE_AUDIO_TRACK_USES_WEB_AUDIO: boolean;
    /**
     * Local Audio Track uses Web Audio (for play)
     * 优先级:  LOCAL_AUDIO_TRACK_USES_WEB_AUDIO < config 中的 bypassWebAudio < DISABLE_WEBAUDIO
     **/
    LOCAL_AUDIO_TRACK_USES_WEB_AUDIO: boolean;
    /** bitrate adapter type (STANDARD_BITRATE / COMPATIBLE_BITRATE / DEFAULT_BITRATE) */
    BITRATE_ADAPTER_TYPE: string;
    /** For AI Denoiser */
    AI_DENOISER_PARAMETERS: {
        excludedLinks: any[];
    };
    /** Audjust 3A from plugins` configs */
    ADJUST_3A_FROM_PLUGINS: boolean;
    RAISE_H264_BASELINE_PRIORITY: boolean;
    FILTER_SEND_H264_BASELINE: boolean;
    /** 用于修改发给网关h264 profile的顺序 */
    FIRST_H264_PROFILE_LEVEL_ID: string;
    /** 用于修改发给网关h264 packetization-mode 的顺序 */
    FIRST_PACKETIZATION_MODE: string;
    /** 可配置的x-google-start-bitrate*/
    X_GOOGLE_START_BITRATE: any;
    NEW_REPORT_SERVER: boolean;
    NEW_REPORT_SERVER_DOMAINS: string[];
    VIDEO_INSPECT_WORKER_MESSAGE_LENGTH_LIMIT: number;
    VIDEO_INSPECT_INTERVAL_MINIMUM: number;
    VIDEO_INSPECT_QUALITY_RATIO: number;
    VIDEO_INSPECT_WORKER_MANAGER_HOST: string;
    VIDEO_INSPECT_WORKER_MANAGER_PORT: string;
    VIDEO_INSPECT_WORKER_PORT: string;
    SHOW_VIDEO_INSPECT_WORKER_MESSAGE: boolean;
    STATS_COLLECTOR_PORT: number;
    FORCE_TURN_TCP: boolean;
    /**
     * WebAudio 初始化参数
     */
    WEBAUDIO_INIT_OPTIONS: any;
    /**
     * 是否过滤视频fec rtpmap
     * */
    FILTER_VIDEO_FEC: boolean;
    /**
     * 是否过滤音频fec rtpmap
     * */
    FILTER_AUDIO_FEC: boolean;
    /**
     * Chrome上的大小流使用encoding
     * */
    CHROME_DUAL_STREAM_USE_ENCODING: boolean;
    DISABLE_DUAL_STREAM_USE_ENCODING: boolean;
    /**
     * 插件用量上报间隔ms
     * */
    EXTENSION_USAGE_UPLOAD_INTERVAL: number;
    /**
     * 是否开启ICE RESTART
     * */
    ICE_RESTART: boolean;
    /**
     * ICE RESTART间隔
     * */
    ICE_RESTART_INTERVAL: number;
    /**
     * 新版本restart ice, 支持tcp/relay
     */
    NEW_ICE_RESTART: boolean;
    /**
     * datachannel 补充参数
     */
    TRANSMITTER_INITIAL_RTT: number;
    TRANSMITTER_INITIAL_RTO: number;
    TRANSMITTER_MAX_BATCH_ACK_COUNT: number;
    TRANSMITTER_MAX_RTO: number;
    DATACHANNEL_COMPRESS: boolean;
    FINGERPRINT: any;
    /** dc join 方法自动 failback 到 websocket 的检测时间 */
    DC_JOIN_WITH_FAILBACK: number;
    /**
     * 是否开启requestVideoFrameCallback方式监测卡顿率
     * */
    ENABLE_VIDEO_FRAME_CALLBACK: boolean;
    /**
     * 视频 xx ms卡顿率
     * */
    VIDEO_FREEZE_DURATION: number;
    /** For Spatial Audio */
    SPATIALIZER_PARAMETERS: {};
    /**日志上传间隔*/
    UPLOAD_LOG_INTERVAL: number;
    /**日志上传初始重试timeout时长*/
    UPLOAD_LOG_REQUEST_RETRY_INTERVAL: number;
    /**日志上传最大重试timeout时长*/
    UPLOAD_LOG_REQUEST_MAX_RETRY_INTERVAL: number;
    /**日志上传关闭后，下次尝试上传间隔*/
    UPLOAD_LOG_TRY_INTERVAL_WHILE_OFF: number;
    UPLOAD_LOG_RETRY_INTERVAL_V1: number;
    UPLOAD_LOG_TWICE_RETRY_INTERVAL_V1: number;
    UPLOAD_LOG_LENGTH_EACH_TIME: number;
    APP_TYPE: number;
    DISABLE_WEBAUDIO: boolean;
    CHANNEL_MEDIA_RELAY_SERVERS: {
        address: string;
        wss: number;
    }[] | undefined;
    KEEP_LAST_FRAME: boolean;
    FORWARD_P2P_CREATION: boolean;
    SYNC_GROUP: boolean;
    /**
     * 是否屏蔽本地client
     */
    BLOCK_LOCAL_CLIENT: boolean;
    /**是否开启ap划分区域, for test */
    AP_AREA: boolean;
    ENABLE_ENCODED_TRANSFORM: boolean;
    ENABLE_VIDEO_SEI: boolean;
    IMAGE_MODERATION_WORKER_HOST: string;
    IMAGE_MODERATION_WORKER_MESSAGE_LENGTH_LIMIT: number;
    IMAGE_MODERATION_INTERVAL_MINIMUM: number;
    SHOW_IMAGE_MODERATION_WORKER_MESSAGE: boolean;
    IMAGE_MODERATION_QUALITY_RATIO: number;
    IMAGE_MODERATION_UPLOAD_REPORT_INTERVAL: number;
    /** 全局AgoraRTC下展示client list */
    SHOW_GLOBAL_CLIENT_LIST: boolean;
    /**
     * datastream 补充参数
     */
    DATASTREAM_MAX_RETRANSMITS: number;
    /**
     * 新版本restart ice，tcp档只是用tcp candidate
     */
    TCP_CANDIDATE_ONLY: boolean;
    EXTERNAL_SIGNAL_REQUEST_TIMEOUT: number;
    /** 展示P2P日志 */
    SHOW_P2P_LOG: boolean;
    /**
     * p2p鉴权超时30s，重置链路
     */
    MAX_P2P_TIMEOUT: number;
    /**
     * p2p鉴权 1s 一次
     */
    P2P_TOKEN_INTERVAL: number;
    /**
     * 展示datastream2 日志
     */
    SHOW_DATASTREAM2_LOG: boolean;
    /**
     * 浏览器切换播放设备限制：Firefox有此接口，但会出现回声
     */
    RESTRICTION_SET_PLAYBACK_DEVICE: boolean;
    /**
     * 适配跨频道连麦端到端加密
     * 开启后，密钥和channelName解绑
     */
    USE_PURE_ENCRYPTION_MASTER_KEY: boolean;
    /**
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * !!!!!!         NOT USED BELOW        !!!!!!
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * */
    ACCOUNT_REGISTER_RETRY_TIMEOUT: number;
    ACCOUNT_REGISTER_RETRY_RATIO: number;
    ACCOUNT_REGISTER_RETRY_TIMEOUT_MAX: number;
    ACCOUNT_REGISTER_RETRY_COUNT_MAX: number;
    AUDIO_CONTEXT: any;
    WEBCS_BACKUP_CONNECT_TIMEOUT: number;
    PLAYER_STATE_DEFER: number;
    SIGNAL_REQUEST_WATCH_INTERVAL: number;
    FILEPATH_LENMAX: number;
    DUALSTREAM_OPERATION_CHECK: boolean;
    MEDIA_ELEMENT_EXISTS_DEPTH: number;
    SHIM_CANDIDATE: boolean;
    LEAVE_MSG_TIMEOUT: number;
    STATS_FILTER: {
        transportId: boolean;
        googTrackId: boolean;
    };
    FILTER_VIDEO_CODEC: any[];
    USE_NEW_NETWORK_CONFIG: boolean;
    AUTO_RESET_AUDIO_ROUTE: boolean;
    PLUGIN_INFO: any[];
    OVERUSE_DETECTOR_PARAMS: {
        /**
         * frameRate adaptation threshold
         */
        MIN_FRAME_RATE: number;
        MAX_FRAME_RATE: number;
        MAX_THRESHOLD_FRAMERATE: number;
        /**
         * bitrateMax/bitrateMin threshold
         */
        BITRATE_MIN_THRESHOLD: number;
        BITRATE_MAX_THRESHOLD: number;
        /**
         * max scale threshold
         */
        MAX_SCALE: number;
        /**
         * bandwidth scale up/down threshold
         */
        BWE_SCALE_UP_THRESHOLD: number;
        BWE_SCALE_DOWN_THRESHOLD: number;
        /**
         * performance scale up/down threshold (sendFrameRate)
         */
        PERF_SCALE_UP_THRESHOLD: number;
        PERF_SCALE_DOWN_THRESHOLD: number;
        /**
         * motion adaptation factor
         */
        MOTION_RESOLUTION_FACTOR: number;
        MOTION_BITRATE_FACTOR: number;
        /**
         * detail adaptation factor
         */
        DETAIL_FRAMERATE_FACTOR: number;
        DETAIL_BITRATE_FACTOR: number;
        /**
         * balance adaptation factor
         */
        BALANCE_RESOLUTION_FACTOR: number;
        BALANCE_FRAMERATE_FACTOR: number;
        BALANCE_BITRATE_FACTOR: number;
        /**
         * detector times threshold
         */
        OVERUSE_TIMES_THRESHOLD: number;
        UNDERUSE_TIMES_THRESHOLD: number;
    };
    ENABLE_AG_ADAPTATION: boolean;
    FORCE_AG_HIGH_FRAMERATE: boolean;
    FORCE_SUPPORT_AG_ADAPTATION: boolean;
    CUSTOM_ADAPTATION_DEFAULT_MODE: string;
    ENCODER_CONFIG_LIMIT: {};
    CAMERA_CAPTURE_CONFIG: any;
    HIDE_NO_POSTER: boolean;
    AP_CACHE_NUM: number;
    AP_UPDATE_INTERVAL: number;
    AP_CACHE_LIFETIME: number;
    MAX_PRELOAD_ASYNC_LENGTH: number;
    ENABLE_PRELOAD: boolean;
    DISABLE_SCREEN_SHARE_REMB: boolean;
};

export declare const MUTABLE_PARAMS_LOCAL_CACHE: MutableParamsCache;

export declare type MutableParamsCache = {
    [key in keyof PartialMutableParams]: {
        value: any;
        expires?: number;
    };
};

export declare interface MutableParamsConfig extends PartialMutableParams {
    __expires?: number;
    __priority?: number;
}

export declare enum NETWORK_INDICATOR_EVENTS {
    NETWORK_STATE_CHANGE = "NETWORK_STATE_CHANGE",
    ONLINE = "ONLINE",
    OFFLINE = "OFFLINE"
}

export declare enum NETWORK_STATE {
    ONLINE = "ONLINE",
    OFFLINE = "OFFLINE"
}

declare class NetworkIndicator extends EventEmitter {
    private _moduleName;
    private _networkState;
    onlineWaiter?: Promise<string>;
    set networkState(state: NETWORK_STATE);
    get networkState(): NETWORK_STATE;
    get isOnline(): boolean;
    constructor();
}

export declare const networkIndicator: NetworkIndicator;

/**
 * 上下行 last mile 网络质量。
 *
 * last mile 是指设备到声网边缘服务器的网络连接。
 *
 * - 本地用户加入频道后，SDK 会每两秒触发一次 [AgoraRTCClient.on("network-quality")]{@link IAgoraRTCClient.event_network_quality} 回调报告本地用户的上下行网络质量。
 * - 可以调用 [AgoraRTCClient.getRemoteNetworkQuality]{@link IAgoraRTCClient.getRemoteNetworkQuality} 方法获取本地订阅的所有远端主播的上下行网络质量。
 *
 * > 该接口报告的网络质量是一个相对值，仅供参考。
 * @public
 */
/** @en
 * The last-mile network quality.
 *
 * Last mile refers to the connection between the local device and Agora edge server.
 *
 * - After the local user joins the channel, the SDK triggers the [AgoraRTCClient.on("network-quality")]{@link IAgoraRTCClient.event_network_quality} callback once every two seconds and provides the uplink and downlink last-mile network conditions of the user through this interface.
 * - You can call [AgoraRTCClient.getRemoteNetworkQuality]{@link IAgoraRTCClient.getRemoteNetworkQuality} to get the network quality of all remote users to whom the local user subscribes.
 *
 * > The returned network quality is a relative value and is for reference only.
 */
export declare interface NetworkQuality {
    /**
     * 上行网络质量。
     *
     * 基于上行发送码率、上行丢包率、平均往返时延和网络抖动计算。
     *
     *  - 0: 质量未知。
     *  - 1: 质量极好。
     *  - 2: 用户主观感觉和极好差不多，但码率可能略低于极好。
     *  - 3: 用户主观感受有瑕疵但不影响沟通。
     *  - 4: 勉强能沟通但不顺畅。
     *  - 5: 网络质量非常差，基本不能沟通。
     *  - 6: 网络连接断开，完全无法沟通。
     */
    /** @en
     * The uplink network quality.
     *
     * It is calculated based on the uplink transmission bitrate, uplink packet loss rate, RTT (round-trip time) and jitter.
     *
     * - 0: The quality is unknown.
     * - 1: The quality is excellent.
     * - 2: The quality is good, but the bitrate is less than optimal.
     * - 3: Users experience slightly impaired communication.
     * - 4: Users can communicate with each other, but not very smoothly.
     * - 5: The quality is so poor that users can barely communicate.
     * - 6: The network is disconnected and users cannot communicate.
     */
    uplinkNetworkQuality: 0 | 1 | 2 | 3 | 4 | 5 | 6;
    /**
     * 下行网络质量。
     *
     * 基于下行发送码率、下行丢包率、平均往返时延和网络抖动计算。
     *
     *  - 0：质量未知。
     *  - 1：质量极好。
     *  - 2：用户主观感觉和极好差不多，但码率可能略低于极好。
     *  - 3：用户主观感受有瑕疵但不影响沟通。
     *  - 4：勉强能沟通但不顺畅。
     *  - 5：网络质量非常差，基本不能沟通。
     *  - 6: 网络连接断开，完全无法沟通。
     */
    /** @en
     * The downlink network quality.
     *
     * It is calculated based on the uplink transmission bitrate, uplink packet loss rate, RTT (round-trip time) and jitter.
     *
     * - 0: The quality is unknown.
     * - 1: The quality is excellent.
     * - 2: The quality is good, but the bitrate is less than optimal.
     * - 3: Users experience slightly impaired communication.
     * - 4: Users can communicate with each other, but not very smoothly.
     * - 5: The quality is so poor that users can barely communicate.
     * - 6: The network is disconnected and users cannot communicate.
     */
    downlinkNetworkQuality: 0 | 1 | 2 | 3 | 4 | 5 | 6;
}

export declare function nextTick(func: () => any): void;

export declare const noop: () => void;

export declare const OVERUSE_DETECTOR_PARAMS: {
    /**
     * frameRate adaptation threshold
     */
    MIN_FRAME_RATE: number;
    MAX_FRAME_RATE: number;
    MAX_THRESHOLD_FRAMERATE: number;
    /**
     * bitrateMax/bitrateMin threshold
     */
    BITRATE_MIN_THRESHOLD: number;
    BITRATE_MAX_THRESHOLD: number;
    /**
     * max scale threshold
     */
    MAX_SCALE: number;
    /**
     * bandwidth scale up/down threshold
     */
    BWE_SCALE_UP_THRESHOLD: number;
    BWE_SCALE_DOWN_THRESHOLD: number;
    /**
     * performance scale up/down threshold (sendFrameRate)
     */
    PERF_SCALE_UP_THRESHOLD: number;
    PERF_SCALE_DOWN_THRESHOLD: number;
    /**
     * motion adaptation factor
     */
    MOTION_RESOLUTION_FACTOR: number;
    MOTION_BITRATE_FACTOR: number;
    /**
     * detail adaptation factor
     */
    DETAIL_FRAMERATE_FACTOR: number;
    DETAIL_BITRATE_FACTOR: number;
    /**
     * balance adaptation factor
     */
    BALANCE_RESOLUTION_FACTOR: number;
    BALANCE_FRAMERATE_FACTOR: number;
    BALANCE_BITRATE_FACTOR: number;
    /**
     * detector times threshold
     */
    OVERUSE_TIMES_THRESHOLD: number;
    UNDERUSE_TIMES_THRESHOLD: number;
};

export declare enum P2PTransportType {
    Default = "default",
    Auto = "auto",
    Relay = "relay",
    SdRtn = "sd-rtn"
}

export declare function parseSdp(sdp: string): SessionDescription;

export declare type PartialMutableParams = Partial<typeof MUTABLE_PARAMS>;

export declare enum PeerConnectionState {
    "new" = 0,
    "connecting" = 1,
    "connected" = 2,
    "disconnected" = 3,
    "failed" = 4,
    "closed" = 5
}

/**
 * 只可能发生 4 种错误
 * - OPERATION_ABORTED 请求操作中止
 * - NETWORK_TIMEOUT 请求超时
 * - NETWORK_RESPONSE_ERROR 响应错误，一般是状态码非法
 * - NETWORK_ERROR 无法定位的网络错误
 */
export declare function post<T>(url: string, options: AxiosRequestConfig, notJson: boolean, withHeaders: true): Promise<{
    data: T;
    headers: any;
}>;

export declare function post<T>(url: string, options: AxiosRequestConfig, notJson?: boolean, withHeaders?: false): Promise<T>;

/**
 * 使用 Protobuf 数据发起 post 请求。
 */
export declare function postProtobuf<T>(url: string, options: AxiosRequestConfig): Promise<T>;

export declare function printSdp(sdp: SessionDescription, EOL?: string): string;

export declare class PromiseMutex {
    static setLogger(_logger: any): void;
    private id;
    private lockingPromise;
    private locks;
    private name;
    private lockId;
    constructor(name?: string, id?: number | string);
    private logger;
    setId(id: number | string): void;
    get isLocked(): boolean;
    lock(info?: string): Promise<() => void>;
}

export declare enum QualityLimitationReason {
    BANDWIDTH = "bandwidth",
    CPU = "cpu",
    NONE = "none",
    OTHER = "other"
}

declare interface ReceiverStats {
    /**
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * !!!!!!       以下都是WebRTC-Stats标准字段       !!!!!!
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * https://w3c.github.io/webrtc-stats/
     * */
    retransmittedBytesReceived?: number;
    retransmittedPacketsReceived?: number;
}

export declare function recursiveMerge(target: unknown, source: unknown): unknown;

export declare function releaseStream(stream: MediaStream): void;

export declare function removeItemFromList<T>(list: T[], item: T): void;

export declare type RequiredOnlyOneOf<T, K extends keyof T = keyof T> = {
    [P in K]: {
        [S in P]: T[S];
    } & {
        [U in keyof Omit<T, P>]?: never;
    };
}[K] extends infer O ? {
    [K in keyof O]: O[K];
} : never;

/**
 * 重置通过 post 统计到的字节数
 */
export declare function resetHTTPBytesCount(): void;

/**
 * 一个基于简单退避的 retry wrapper
 * 仅当 Promise resolve/reject 时会触发 retry，此时 wrapper 会将 result/error 传入 @param handleResult/@param handleError
 * 如果 @param handleResult/@param handleError 返回 true 就会触发退避重试策略，否则该函数直接 resolve/reject 这个 result/error
 */
export declare function retryable<T, E = any>(operation: () => Promise<T>, handleResult?: (result: T, count: number) => boolean, handleError?: (error: E, count: number) => boolean, config?: Partial<RetryConfiguration>): CancelablePromise<T>;

/**
 * @ignore
 * 控制 SDK 重连策略，用于 [createClient]{@link createClient} 时传入的客户端配置。
 *
 * SDK 的重连策略如下:
 * - `timeout` 为连接出现错误开始到下次重试的等待时间，默认为 500 ms。
 * - 当请求第一次出现错误时，等待 `timeout` 时间后再次重试。
 * - 当再次重试仍然出现错误时，更新 `timeout` 的值，设 `timeoutFactor` 为更新系数，令 `timeout` = `timeoutFactor` × `timeout`，等待新的 `timeout` 时间后再次重试。
 * - `maxRetryTimeout` 为 `timeout` 可能更新到的最大值，每次 `timeout` 更新时，如果大于这个值，都会被强制设置为这个值。默认是 10000 ms，也就是最多等待 10 秒后重试。
 * - `maxRetryCount` 为可连续重试的最大次数，当重试次数超过这个值后还是错误，就会停止重试直接抛出错误，默认为 `Infinity`，也就是会无限重试。
 *
 * 你可以通过改动这 4 个参数自定义重连策略。
 */
/** @en
 * @ignore
 */
export declare interface RetryConfiguration {
    /**
     * 重试间隔时间，默认值 500，单位 ms。
     */
    timeout: number;
    /**
     * 重试间隔时间更新系数。
     *
     * 默认值 1.5。每次重试后，都会将 {@link timeout} 乘以该值更新重试间隔时间。
     */
    timeoutFactor: number;
    /**
     * 允许最大的重试次数。
     *
     * 如果连续失败重试超过该值就会停止重试，触发相应的错误。默认值为 `Infinity`，也就是无限重试。
     */
    maxRetryCount: number;
    /**
     * 允许最大的重试间隔时间。
     *
     * 该值指定重试间隔时间 {@link timeout} 的上限。每次 `timeout` 更新时，如果大于这个值，都会被强制设置为这个值。默认为 10000 ms，也就是最多等待 10 秒后重试。
     */
    maxRetryTimeout: number;
}

/**
 * 最基础的移动平均实现
 */
export declare class Rolling {
    private input;
    private size;
    constructor(size: number);
    add(num: number): void;
    mean(): number;
}

export declare interface RTCLegacyStatsObject {
    id: string;
    timestamp: string;
    type: string;
    [name: string]: string;
}

export declare interface RTCLegacyStatsReport {
    names: () => string[];
    stat: (name: string) => string;
    id: string;
    timestamp: Date;
    type: string;
}

declare interface RTCStats_2 {
    timestamp: DOMHighResTimeStamp;
    type: RTCStatsType_2;
    id: string;
    [name: string]: any;
}
export { RTCStats_2 as RTCStats }

declare type RTCStatsReport_2 = Map<string, RTCStats_2>;
export { RTCStatsReport_2 as RTCStatsReport }

export declare interface RTCStatsResponse {
    result: () => RTCLegacyStatsReport[];
}

declare enum RTCStatsType_2 {
    CERTIFICATE = "certificate",
    CODEC = "codec",
    CANDIDATE_PAIR = "candidate-pair",
    LOCAL_CANDIDATE = "local-candidate",
    REMOTE_CANDIDATE = "remote-candidate",
    INBOUND = "inbound-rtp",
    TRACK = "track",
    OUTBOUND = "outbound-rtp",
    PC = "peer-connection",
    REMOTE_INBOUND = "remote-inbound-rtp",
    REMOTE_OUTBOUND = "remote-outbound-rtp",
    TRANSPORT = "transport",
    CSRC = "csrc",
    DATA_CHANNEL = "data-channel",
    STREAM = "stream",
    SENDER = "sender",
    RECEIVER = "receiver"
}
export { RTCStatsType_2 as RTCStatsType }

export declare interface RTMConfiguration {
    /** RTM团队透传字段给ap */
    apRTM: boolean;
    rtmFlag: number;
}

/**
 * 传入一个 key 然后执行一个函数，对于相同的 key，这个函数只会被执行一次
 */
export declare function runOnce(func: () => void, key: string): void;

export declare function safeCloneJson<T>(obj: T): T;

export declare enum ScalabilityMode {
    L1T1 = "L1T1",
    L1T2 = "L1T2",
    L1T3 = "L1T3",
    L1T3_KEY = "L1T3_KEY",
    L2T1_KEY = "L2T1_KEY",
    L2T2_KEY = "L2T2_KEY",
    L2T3_KEY = "L2T3_KEY",
    L3T1_KEY = "L3T1_KEY",
    L3T2_KEY = "L3T2_KEY",
    L3T3_KEY = "L3T3_KEY"
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
export declare type SDK_AUDIO_CODEC = keyof typeof AudioCodec;

/**
 * 浏览器使用的编码格式，有以下两种选择：
 * - `"vp8"`: 浏览器使用 VP8 编码。
 * - `"h264"`: 浏览器使用 H.264 编码。
 * - `"vp9"`: (Beta) 浏览器使用 VP9 编码。
 * - `"av1"`: 浏览器使用 AV1 编码。
 *
 * > Safari 12.1 之前版本不支持 VP8 编码。
 * @public
 */
/** @en
 * The codec that the Web browser uses for encoding.
 * - `"vp8"`: Use VP8 for encoding.
 * - `"h264"`: Use H.264 for encoding.
 * - `"vp9"`: (Beta) Use VP9 for encoding.
 * - `"av1"`: Use AV1 for encoding.
 *
 * > Safari 12.1 or earlier does not support the VP8 codec.
 */
export declare type SDK_CODEC = keyof typeof VideoCodec;

/**
 * 频道场景。
 *
 * Agora Web SDK 需知道 app 的使用场景（例如通信模式或直播模式），从而使用不同的优化手段。
 *
 * 声网频道支持以下场景:
 * - `"live"`: 直播场景，有主播和观众两种用户角色，可以通过 [setClientRole]{@link IAgoraRTCClient.setClientRole} 方法设置主播和观众的角色。主播可以收发音视频流，而观众只能接收音视频流，无法发送。
 * - `"rtc"`: 通信场景，用于常见的一对一通话或群聊，频道中的任何用户可以自由说话。
 * @public
 */
/** @en
 * The channel profile.
 *
 * The SDK differentiates channel profiles and applies different optimization algorithms accordingly. For example, it prioritizes smoothness and low latency for a video call, and prioritizes video quality for a video broadcast.
 *
 * The SDK supports the following channel profiles:
 * - `"live"`: Sets the channel profile as live broadcast. You need to go on to call [setClientRole]{@link IAgoraRTCClient.setClientRole} to set the client as either a host or an audience. A host can send and receive audio or video, while an audience can only receive audio or video.
 * - `"rtc"`: Sets the channel profile as communication. It is used for a one-on-one call or a group call where all users in the channel can converse freely.
 */
export declare type SDK_MODE = "live" | "rtc" | "p2p";

/**
 * 主要存储状态日志相关的信息
 * 期望可以通过redux develop tool能够描述出一些SDK内部状态的流转
 * */
export declare class SDKStore {
    private state;
    constructor(codec: SDK_CODEC, audioCodec: SDK_AUDIO_CODEC, mode: SDK_MODE, clientId: string);
    private dispatch;
    set sessionId(sessionId: string | null);
    get sessionId(): string | null;
    set codec(codec: SDK_CODEC);
    get codec(): SDK_CODEC;
    get mode(): SDK_MODE;
    get audioCodec(): SDK_AUDIO_CODEC;
    get clientId(): string;
    set p2pId(p2pId: number);
    get p2pId(): number;
    set dcId(dcId: number);
    get dcId(): number;
    set uid(uid: UID | undefined);
    get uid(): UID | undefined;
    set intUid(intUid: number | undefined);
    get intUid(): number | undefined;
    set pubId(pubId: number);
    get pubId(): number;
    set cloudProxyServerMode(mode: CloudProxyServerMode);
    get cloudProxyServerMode(): CloudProxyServerMode;
    set useP2P(val: boolean);
    get useP2P(): boolean;
    set p2pTransport(val: P2PTransportType);
    get p2pTransport(): P2PTransportType;
    clientCreated(): void;
    joinStart(): void;
    joinEnd(): void;
    requestAPStart(): void;
    requestAPEnd(): void;
    joinGatewayStart(): void;
    joinGatewayEnd(): void;
    peerConnectionStart(): void;
    peerConnectionEnd(): void;
    descriptionStart(): void;
    signalChannelOpen(): void;
    iceConnectionEnd(): void;
    publish(trackId: string, type: "video" | "audio" | "datachannel", publishStart?: number, publishEnd?: number): void;
    subscribe(userId: UID, type: "video" | "audio", subscribeStart?: number, subscribeEnd?: number, firstFrame?: number, streamAdded?: number, firstDecoded?: number): void;
    massSubscribe(userList: {
        userId: UID;
        type: "video" | "audio";
    }[], subscribeStart?: number, subscribeEnd?: number, firstFrame?: number): void;
    get keyMetrics(): KeyMetrics;
    recordJoinChannelService(record: Partial<JoinChannelServiceRecord>, index?: number): number;
    resetJoinChannelServiceRecords(): void;
    resetKeyMetrics(): void;
    get joinChannelServiceRecords(): JoinChannelServiceRecord[];
    get avoidJoinStart(): number;
    set avoidJoinStart(avoidJoinStart: number);
}

export declare interface SelectedCandidatePairStats {
    localCandidate: CandidateStats;
    remoteCandidate: CandidateStats;
    id: string;
}

declare interface SenderStats {
    /**
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * !!!!!!       以下都是WebRTC-Stats标准字段       !!!!!!
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * https://w3c.github.io/webrtc-stats/
     * */
    /**
     * The total number of bytes that were retransmitted for this SSRC, only including payload bytes. (不包含header)
     * This is a subset of bytesSent.
     * If RTX is not negotiated, retransmitted bytes are sent over this ssrc.
     * If RTX was negotiated, retransmitted bytes are sent over a separate SSRC but is still accounted for here.
     */
    retransmittedBytesSent?: number;
    retransmittedPacketsSent?: number;
}

export declare function setBigInt64(dataView: DataView, byteOffset: number, value: bigint, littleEndian: boolean | undefined): void;

export declare function setBigUint64(dataView: DataView, byteOffset: number, value: bigint, littleEndian?: boolean): void;

/**
 * 设置可变的全局变量
 * @param key - 全局变量的 key
 * @param value - 全局变量的值，可以为任意对象
 * @param _force - 在配置下发有相同key的情况下是否强制设置
 * @internal
 */
export declare function setParameter(key: keyof typeof MUTABLE_PARAMS, value: any, _force?: boolean): void;

export declare const SHA256: (text: string) => Promise<string>;

export declare function shiftAndPush(data: number, length?: number): number[];

/**
 * 显示一个界面展示所有的 sources，当用户完成选择时，resolve 目标的 sourceId
 */
export declare function showElectronSelectSourceWindow(sources: ElectronDesktopCapturerSource[], captureAudio?: boolean): Promise<{
    sourceId: string;
    audio: boolean;
}>;

export declare interface StatsFilterOptions {
    updateInterval: number;
    /** 丢包率统计窗口间隔，默认为 2s */
    lossRateInterval: number;
    /** 卡顿率统计窗口间隔 默认 500 ms */
    freezeRateLimit: number;
    /** 视频解码超时上报timeout 默认 10*1000 ms */
    firstVideoDecodedTimeout: number;
}

export declare interface StreamInfo {
    audio: boolean;
    video: boolean;
    muteAudio: boolean;
    muteVideo: boolean;
}

export declare const supportIsSecureContext: () => boolean;

export declare const throttleByKey: any;

/**
 * 将在 timeout 时间后 reject 一个 TIMEOUT 错误
 */
export declare function timeout<T>(timeout: number): Promise<T>;

export declare function toCamelCase(snakeStr: string): string;

/**
 * 将 3.0.0-alpha.19 / 4.20.1-1 这种版本号转为合法的 4 位版本号
 *
 * 版本号规范：
 * - 4.20.1-1 表示 hotfix 版本，占第 4 位版本号的【0-100），所以 4.20.1-1 -> 4.20.1.1
 * - 4.20.1-100 表示 special 版本，占第 4 位版本号的 100-100 * n，所以 4.20.1-100 -> 4.20.1.100
 * - 如果没有版本前缀，代表正式版本，直接返回 3 位版本号
 */
export declare function transferVersion(version: string): string;

export declare function transformBrowserName(result: UAParser_2.IResult): string;

export declare function transformBrowserVersion(result: UAParser_2.IResult, isMajor?: boolean): string;

export declare function transformOSName(result: UAParser_2.IResult): string;

export declare class TransformStreamAdapter<T> {
    private _transformStream;
    private _readable;
    private _writable;
    private _reader;
    private _writer;
    constructor(transform?: TransformerTransformCallback<T, any>);
    get reader(): ReadableStreamDefaultReader<T>;
    get writer(): WritableStreamDefaultWriter<T>;
    close(): void;
}

export declare interface TransportStats {
    connectState: RTCPeerConnectionState;
}

/**
 * @ignore
 * TURN 服务器配置，在调用 [setTurnServer]{@link IAgoraRTCClient.setTurnServer} 时使用。
 * @public
 */
/** @en
 * @ignore
 * The configuration of your TURN server. Used when calling [setTurnServer]{@link IAgoraRTCClient.setTurnServer}.
 */
export declare interface TurnServerConfig {
    /**
     * TURN 服务器 URL 地址，ASCII 字符。
     */
    /** @en
     * The URL of your TURN server. ASCII characters only.
     */
    turnServerURL: string;
    /**
     * 在 TURN 服务器上使用的密码，ASCII 字符。
     */
    /** @en
     * The password of Your TURN server. ASCII characters only.
     */
    password: string;
    /**
     * 想要添加的 UDP 端口。
     */
    /** @en
     * The TCP port(s) you want add to your TURN server.
     */
    udpport?: number;
    /**
     * 在 TURN 服务器上注册并使用的用户名，ASCII 字符。
     */
    /** @en
     * The username of your TURN server. ASCII characters only.
     */
    username: string;
    /**
     * 是否启用强制中转：
     * - `true`: 强制所有流由 TURN 服务器进行中转。
     * - `false`: （默认）不强制所有流由 TURN 服务器进行中转。
     */
    /** @en
     * Whether to force data transfer by the TURN Server:
     * - `true`: Force data transfer.
     * - `false`: (default) Do not force data transfer.
     */
    forceturn?: boolean;
    /**
     * 想要添加的 TCP 端口。
     */
    /** @en
     * The UDP port(s) you want to add to your TURN server.
     */
    tcpport?: number;
    security?: boolean;
}

export declare interface TurnServerConfigWithMode {
    mode: TurnServerMode;
    servers: TurnServerConfig[] | RTCIceServer[];
    serversFromGateway?: TurnServerConfig[];
}

/**
 * - original-manual 用户设置了原生的 iceServers 列表.
 * - manual 用户通过代理设置了 turn server
 * - auto 用户没有走代理
 * - off 不使用 TurnServer
 */
export declare type TurnServerMode = "auto" | "manual" | "off" | "original-manual";

/**
 * 用于标识用户的 ID。同一频道中每个用户都具有唯一的 ID，且数据类型相同。
 *
 * 为保证最佳的用户体验，声网强烈建议你不要使用字符串作为用户 ID。详见 {@link join} 方法中的说明。
 * @public
 */
/** @en
 * The user ID to identify a user in the channel.
 *
 * Each user in the same channel should have a unique user ID with the same data type (number or string).
 *
 * To ensure a better end-user experience, Agora recommends using a number as the user ID. See {@link join} for details.
 */
export declare type UID = number | string;

export declare function uint8ArrayConcat(...arrays: Uint8Array[]): Uint8Array;

export declare function uint8ArrayToBase64(array: Uint8Array): string;

export declare function updateConcurrentNumber(uuid: string, num: number): void;

export declare const VERSION: string;

export declare enum VideoCodec {
    h264 = "h264",
    h265 = "h265",
    vp8 = "vp8",
    vp9 = "vp9",
    av1 = "av1"
}

export declare interface VideoFrameState {
    width: number;
    height: number;
    frameRate: number;
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare enum VisibleHiddenReason {
    COVERED = "COVERED",
    POSITION = "POSITION",
    SIZE = "SIZE",
    STYLE = "STYLE"
}

declare enum VisibleHiddenReasonInner {
    UNMOUNTED = "UNMOUNTED",
    INVALID_HTML_ELEMENT = "INVALID_HTML_ELEMENT"
}

export declare interface VisibleHiddenResult {
    visible: false;
    reason: keyof typeof VisibleHiddenReason;
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare interface VisibleHiddenResultInner {
    visible: false;
    reason: keyof typeof VisibleHiddenReason | keyof typeof VisibleHiddenReasonInner;
}

/**
 * @ignore
 */
/** @en
 * @ignore
 */
declare interface VisibleResultInner {
    visible: true;
}

/**
 * 将在 timeout 时间后 resolve
 */
export declare function wait(timeout: number): Promise<void>;

export declare enum WebSocketQuitReason {
    NETWORK_ERROR = "NETWORK_ERROR",
    SERVER_ERROR = "SERVER_ERROR",
    MULTI_IP = "MULTI_IP",
    TIMEOUT = "TIMEOUT",
    OFFLINE = "OFFLINE",
    LEAVE = "LEAVE",
    P2P_FAILED = "P2P_FAILED",
    FALLBACK = "FALLBACK",
    REGIONAL_DISTRIBUTION = "REGIONAL_DISTRIBUTION"
}

/**
 * 如果一个 Promise 没有在 timeout 时间内 reject/resolve，就 reject 一个 TIMEOUT
 */
export declare function withTimeout<T>(promise: Promise<T>, time: number): Promise<T>;


export * from "sdp-parser2";

export { }
